package net.leanix.dropkit.oauth;

import com.fasterxml.jackson.databind.ObjectMapper;
import javax.ws.rs.core.HttpHeaders;
import com.google.common.base.Optional;
import com.google.inject.Inject;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse.Status;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.core.util.Base64;
import io.dropwizard.auth.AuthenticationException;
import io.dropwizard.auth.Authenticator;
import java.io.IOException;
import org.slf4j.Logger;

/**
 * Authenticator used to protect resources.
 *
 *
 */
public class OAuth2Authenticator implements Authenticator<String, AuthenticatedUser> {

    /**
     * Constant for the access token (oauth2 spec).
     */
    private static final String BEARER = "bearer";

    private final String tokenVerificationUrl;
    private final String authorizationValue;
    private final Logger logger;

    private final Client client = Client.create();
    private static final ObjectMapper mapper = new ObjectMapper();

    static {
        mapper.disableDefaultTyping();
    }

    /**
     * Constructor.
     *
     * @param configuration
     * @param logger
     */
    @Inject
    public OAuth2Authenticator(OAuth2ClientConfig configuration, Logger logger) {
        System.setProperty("jsse.enableSNIExtension", "false"); // prevents an ssl handshake error with java 7
        tokenVerificationUrl = configuration.getVerificationUrl();
        authorizationValue = "Basic ".concat(
                new String(Base64.encode(configuration.getClientId().concat(":").concat(configuration.getClientSecret()).getBytes()))
        );
        this.logger = logger;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * com.yammer.dropwizard.auth.Authenticator#authenticate(java.lang.Object)
     */
    @Override
    public Optional<AuthenticatedUser> authenticate(String accessToken) throws AuthenticationException {
        return Optional.fromNullable(this.verify(accessToken));
    }

    private boolean isValidResponse(VerifyTokenResponse tokenResponse) {
        return tokenResponse != null && tokenResponse.getPrincipal() != null && tokenResponse.getError() == null;
    }

    private AuthenticatedUser verify(String accessToken) throws AuthenticationException {
        
        try {
            String json;
            logger.debug("Verifying access token " + accessToken + " against " + tokenVerificationUrl);
            json = client
                    .resource(String.format(tokenVerificationUrl.concat("?access_token=%s"), accessToken))
                    .header(HttpHeaders.AUTHORIZATION, authorizationValue).accept("application/json")
                    .get(String.class);
            
            VerifyTokenResponse response = mapper.readValue(json, VerifyTokenResponse.class);
            
            if (isValidResponse(response)) {
                return response.getPrincipal();
            }

            return null;
            
        } catch (UniformInterfaceException ex) {
            if (ex.getResponse().getStatus() == Status.GONE.getStatusCode()) {
                logger.warn("Access token {} is not valid any more: {}", accessToken, ex.getMessage());
                return null;   
            } else {
                logger.error("Unable to verify token " + accessToken + ": " + ex.getMessage(), ex);
                throw new AuthenticationException("Access token verification failed: " + ex.getMessage(), ex);
            }
        } catch (IOException ex) {
            logger.error("Unable to parse verification response for token " + accessToken + ": " + ex.getMessage(), ex);
            throw new AuthenticationException("Unable to parse verification response: " + ex.getMessage(), ex);
        } catch (Throwable ex) {
            logger.error("Unable to verify token " + accessToken + ", unknown reason: " + ex.getMessage(), ex);
            throw new AuthenticationException("Unable to perform authentication, unknown reason.", ex);
        }
    }
}
