package net.leanix.dropkit.api;

import com.fasterxml.jackson.jaxrs.json.JacksonJsonProvider;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import java.net.URI;
import net.leanix.dropkit.oauth.ClientCredentialAccessTokenFactory;
import net.leanix.dropkit.oauth.ClientCredentialsApiAuthFilter;
import net.leanix.dropkit.oauth.OAuth2ClientConfig;

/**
 * Client factory.
 *
 * Returns a jersey api client with configured with client credential
 * authentication.
 */
public class ClientFactory {

    /**
     * Factory method to create an oauth2 api client used by an instance.
     *
     * @param config
     * @return
     */
    public static Client create(OAuth2ClientConfig config) {
        URI tokenURI = URI.create(config.getTokenUrl());

        com.sun.jersey.api.client.Client c = createJerseyClientWithJacksonSerializer();
        c.setFollowRedirects(Boolean.TRUE);
        c.addFilter(getAuthFilter(ClientCredentialAccessTokenFactory.create(config)));

        return new Client(config.getBaseUrl(), c);
    }

    /**
     * Factory method to create an api client used by an instance.
     *
     * @param baseUrl like https://mtm.leanix.net/api/v1 , without resource
     * path.
     * @param tokenUrl
     * @param verificationUrl
     * @param clientId
     * @param clientSecret
     * @return
     */
    public static Client create(
            final String baseUrl,
            final String tokenUrl,
            final String verificationUrl,
            final String clientId,
            final String clientSecret
    ) {
        OAuth2ClientConfig config = new OAuth2ClientConfig();
        config.setBaseUrl(baseUrl);
        config.setTokenUrl(tokenUrl);
        config.setVerificationUrl(verificationUrl);
        config.setClientId(clientId);
        config.setClientSecret(clientSecret);

        return create(config);
    }

    /**
     * Factory method to create an api client that uses no authentication.
     *
     * @param baseUrl like https://mtm.leanix.net/api/v1 , without resource
     * path.
     * @return
     */
    public static Client create(String baseUrl) {
        com.sun.jersey.api.client.Client c = createJerseyClientWithJacksonSerializer();
        c.setFollowRedirects(Boolean.TRUE);
        return new Client(baseUrl, c);
    }

    /**
     * Creates a jersey client that can read application/json bodies.
     *
     * @return
     */
    public static com.sun.jersey.api.client.Client createJerseyClientWithJacksonSerializer() {
        ClientConfig cc = new DefaultClientConfig();
        cc.getClasses().add(JacksonJsonProvider.class);
        return com.sun.jersey.api.client.Client.create(cc);
    }

    private static ClientCredentialsApiAuthFilter getAuthFilter(ClientCredentialAccessTokenFactory factory) {
        return new ClientCredentialsApiAuthFilter(factory);
    }
}
