package net.leanix.dropkit.oauth;

import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.fasterxml.jackson.datatype.joda.ser.DateTimeSerializer;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import com.wordnik.swagger.annotations.ApiModelProperty;
import net.leanix.dropkit.api.DateTimeDeserializer;
import net.leanix.dropkit.api.Link;
import net.leanix.dropkit.api.LinkProvider;

import org.joda.time.DateTime;
import org.slf4j.LoggerFactory;

/**
 * Permission for a user to access a workspace.
 *
 * Defaults to active contact.
 *
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class Permission implements Serializable, WorkspacePermission, LinkProvider {

    private UUID id;
    private UUID workspaceId;
    private PermissionRole role;
    private PermissionStatus status = PermissionStatus.ACTIVE;
    private DateTime lastLogin;
    private final List<Link> links = new ArrayList<>();

    public Permission() {
        lastLogin = new DateTime();
    }

    public void setId(UUID id) {
        this.id = id;
    }

    @ApiModelProperty(dataType = "string")
    public UUID getId() {
        return id;
    }

    @Override
    @ApiModelProperty(dataType = "string")
    public UUID getWorkspaceId() {
        return workspaceId;
    }

    public void setWorkspaceId(UUID workspaceId) {
        this.workspaceId = workspaceId;
    }

    @Override
    @ApiModelProperty(dataType = "string")
    public PermissionStatus getStatus() {
        return status;
    }

    public void setStatus(PermissionStatus status) {
        this.status = status;
    }

    @JsonSerialize(using = DateTimeSerializer.class)
    @Override
    public DateTime getLastLogin() {
        return lastLogin;
    }

    @JsonDeserialize(using = DateTimeDeserializer.class)
    public void setLastLogin(DateTime lastLogin) {
        this.lastLogin = lastLogin;
    }

    @JsonIgnore
    @Override
    public boolean isActive() {
        return PermissionStatus.ACTIVE.equals(status);
    }

    @JsonIgnore
    @Override
    public boolean isActive(PermissionRole role) {
        return isActive() && hasRole(role);
    }

    public PermissionRole getRole() {
        return role;
    }

    public void setRole(PermissionRole role) {
        this.role = role;
    }

    /**
     * Checks if the permission has the given role.
     *
     * @param role
     * @return
     */
    @Override
    public boolean hasRole(PermissionRole role) {
        return this.role != null && this.role.equals(role);
    }

    /**
     * The HATEOAS relations.
     *
     * @return
     */
    @Override
    @JsonProperty
    public List<Link> getLinks() {
        return links;
    }

    @Override
    public String toString() {
        return "[Permission: Workspace " + workspaceId + ", role " + role + ", status " + status.toString() + "]";
    }
}
