package net.leanix.dropkit.oauth;

import com.fasterxml.jackson.databind.ObjectMapper;
import javax.ws.rs.core.HttpHeaders;
import com.google.common.base.Optional;
import com.google.inject.Inject;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.core.util.Base64;
import io.dropwizard.auth.AuthenticationException;
import io.dropwizard.auth.Authenticator;
import java.io.IOException;
import net.leanix.dropkit.BusinessLogicException;
import org.slf4j.Logger;

/**
 * Authenticator used to protect resources.
 *
 *
 */
public class OAuth2Authenticator implements Authenticator<String, AuthenticatedUser> {

    /**
     * Constant for the access token (oauth2 spec).
     */
    private static final String BEARER = "bearer";

    /**
     * Constant name of the request attribute where the response is stored.
     */
    public static final String VERIFY_TOKEN_RESPONSE = "VERIFY_TOKEN_RESPONSE";

    private final String tokenVerificationUrl;
    private final String authorizationValue;
    private final Logger logger;

    private final Client client = Client.create();
    private static final ObjectMapper mapper = new ObjectMapper();

    static {
        mapper.disableDefaultTyping();
    }

    /**
     * Constructor.
     *
     * @param configuration
     * @param logger
     */
    @Inject
    public OAuth2Authenticator(OAuth2ClientConfig configuration, Logger logger) {
        System.setProperty("jsse.enableSNIExtension", "false"); // prevents an ssl handshake error with java 7
        tokenVerificationUrl = configuration.getVerificationUrl();
        authorizationValue = "Basic ".concat(
                new String(Base64.encode(configuration.getClientId().concat(":").concat(configuration.getClientSecret()).getBytes()))
        );
        this.logger = logger;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * com.yammer.dropwizard.auth.Authenticator#authenticate(java.lang.Object)
     */
    @Override
    public Optional<AuthenticatedUser> authenticate(String accessToken) throws AuthenticationException {
        try {
            VerifyTokenResponse response = this.verify(accessToken);
            return Optional.fromNullable(response.getPrincipal());
        } catch (BusinessLogicException | AuthenticationException | IOException ex) {
            logger.error(ex.getMessage());
        }

        return Optional.fromNullable(null);
    }

    private boolean isValidResponse(VerifyTokenResponse tokenResponse) {
        return tokenResponse != null && tokenResponse.getPrincipal() != null
                && tokenResponse.getError() == null;
    }

    public VerifyTokenResponse verify(String accessToken) throws BusinessLogicException, AuthenticationException, IOException {
        String json;
        try {
            logger.debug("Verifying access token " + accessToken + " against " + tokenVerificationUrl);
            json = client
                    .resource(String.format(tokenVerificationUrl.concat("?access_token=%s"), accessToken))
                    .header(HttpHeaders.AUTHORIZATION, authorizationValue).accept("application/json")
                    .get(String.class);
        } catch (UniformInterfaceException ex) {
            throw new AuthenticationException("Access token verification failed: " + ex.getMessage(), ex);
        }

        final VerifyTokenResponse response;

        response = mapper.readValue(json, VerifyTokenResponse.class);
        if (isValidResponse(response)) {
            logger.debug("Access token verfication response " + accessToken + ": " + response.toString());
            return response;
        }

        return null;
    }
}
