package net.leanix.dropkit.persistence;

import com.google.common.collect.ImmutableList;
import com.google.inject.AbstractModule;
import com.google.inject.Provides;
import io.dropwizard.Configuration;
import io.dropwizard.db.DataSourceFactory;
import io.dropwizard.db.DatabaseConfiguration;
import io.dropwizard.hibernate.HibernateBundle;
import io.dropwizard.hibernate.SessionFactoryFactory;
import io.dropwizard.migrations.DbCommand;
import io.dropwizard.setup.Environment;
import org.hibernate.SessionFactory;

/**
 * Guice module for persistence.
 *
 *
 * @param <T>
 */
public class PersistenceModule<T extends Configuration & DataSourceFactoryProvider> extends AbstractModule implements DatabaseConfiguration<T> {

    private final ImmutableList<Class<?>> bundleEntities;
    private HibernateBundle<T> hibernateBundle;
    private final Class<T> configClass;

    /**
     * Pass the entities to be regarded by the hibernate bundle to the
     * constructor.
     *
     * <code>new PersistenceModule(A.class, B.class, C.class)</code>
     *
     * @param configClass
     * @param entity
     * @param entities
     */
    public PersistenceModule(Class<T> configClass, Class<?> entity, Class<?>... entities) {
        super();
        this.configClass = configClass;
        bundleEntities = ImmutableList.<Class<?>>builder().add(entity).add(entities).build();
    }

    private HibernateBundle getHibernateBundle() {
        if (hibernateBundle == null) {
            hibernateBundle = new HibernateBundle<T>(bundleEntities, new SessionFactoryFactory()) {
                @Override
                public DataSourceFactory getDataSourceFactory(T configuration) {
                    return configuration.getDataSourceFactory();
                }
            };
        }

        return hibernateBundle;
    }

    /**
     * Provides the session factory for DAO objects.
     *
     * @param configuration
     * @param environment
     * @return
     * @throws java.lang.Exception
     */
    @Provides
    public SessionFactory provideSessionFactory(
            DataSourceFactoryProvider configuration,
            Environment environment
    ) throws Exception {
        SessionFactory sf = getHibernateBundle().getSessionFactory();
        if (sf == null) {
            getHibernateBundle().run(configuration, environment);
            return getHibernateBundle().getSessionFactory();
        }

        return sf;
    }

    @Provides
    public DbCommand getMigrationsCommand() {
        return new DbCommand<>(this, configClass);
    }

    @Override
    protected void configure() {
        bind(DataSourceFactoryProvider.class).to(configClass);
    }

    /**
     * This method is required for the migration db command (module implements
     * DatabaseConfiguration).
     *
     * @param t
     * @return
     */
    @Override
    public DataSourceFactory getDataSourceFactory(T t) {
        return t.getDataSourceFactory();
    }

    public Class<T> getConfigClass() {
        return configClass;
    }
}
