package net.leanix.dropkit.persistence;

import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import io.dropwizard.hibernate.AbstractDAO;
import java.util.ArrayList;
import java.util.Iterator;
import org.hibernate.Criteria;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.internal.CriteriaImpl;
import org.hibernate.internal.CriteriaImpl.Subcriteria;

/**
 * BaseDAO
 *
 * @param <T> the entity type
 */
public class BaseDAO<T> extends AbstractDAO<T> {

    public BaseDAO(SessionFactory factory) {
        super(factory);
    }

    /**
     * Returns the count based on the given criteria.
     *
     * @param criteria conditions to meet
     * @return
     */
    public Long count(Criteria criteria) {
        criteria.setProjection(Projections.rowCount());
        return (Long) criteria.uniqueResult();
    }

    /**
     * Generic find method that uses criteria and supports pagination and
     * ordering.
     *
     * @param pageRequest request for paging
     * @param criteria hibernate criteria api restrictions for query
     * @return
     */
    public PageResult<T> find(PageRequest pageRequest, Criteria criteria) {
        PageResult<T> pageResult = new PageResult<>();

        // Get count
        pageResult.setTotal(count(criteria));

        // Get list
        criteria.setProjection(null);
        criteria.setResultTransformer(Criteria.ROOT_ENTITY);
        Criteria curCriteria = this.createCriteria(pageRequest, criteria);
        pageResult.setPageRequest(pageRequest);
        pageResult.setData(curCriteria.list());

        return pageResult;
    }

    /**
     * Generic find method that uses criteria and supports pagination and
     * ordering.
     *
     * @param pageRequest request for paging
     * @return
     */
    public PageResult<T> find(PageRequest pageRequest) {
        return this.find(pageRequest, this.createCriteria());
    }

    /**
     * Returns a query-ready hibernate criteria.
     *
     * @return query-ready hibernate criteria
     */
    public Criteria createCriteria() {
        return criteria();
    }

    /**
     * Returns a query-ready hibernate criteria. Includes pagination, sorting
     * and restrictions.
     *
     * @param pageRequest request for paging
     * @param criteria hibernate criteria restrictions for query
     * @return query-ready hibernate criteria
     */
    public Criteria createCriteria(PageRequest pageRequest, Criteria criteria) {
        // set ordering
        if (pageRequest.hasSort()) {
            for (String stringOrder : pageRequest.getSort().split(",")) {
                addOrder(criteria, stringOrder);
            }
        }

        if (pageRequest.getPage() > 0 && pageRequest.getSize() > 0) {
            criteria.setFirstResult((pageRequest.getPage() - 1) * pageRequest.getSize());
            criteria.setMaxResults(pageRequest.getSize());
        }

        return criteria;
    }

    /**
     * Handles the sort order.
     *
     * @param criteria
     * @param stringOrder
     * @link http://stackoverflow.com/questions/870029/hibernate-order-by-association
     */
    private void addOrder(Criteria criteria, String stringOrder) {
        String direction = stringOrder.toLowerCase().endsWith("desc") ? "desc" : "asc";

        String sortProp = stringOrder.replaceAll("(?i)-" + direction, "");

        //we need to introduce an alias for relations
        if (sortProp.contains(".")) {
            ArrayList<String> parts = Lists.newArrayList(Splitter.on('.').trimResults().omitEmptyStrings().split(sortProp));
            if (parts.size() == 1) {
                return;
            }

            createAliasIfNotAlreadyExist(criteria, parts.get(0), parts.get(0));
        }

        if ("asc".equals(direction)) {
            criteria.addOrder(Order.asc(sortProp));
        } else {
            criteria.addOrder(Order.desc(sortProp));
        }
    }

    /**
     *
     * @link http://taghjijtopentech.blogspot.de/2013/09/hibernate-criteria-duplicate-alias-and.html
     *
     * @param criteria
     * @param associationPath
     * @param newAlias
     * @return
     */
    private String createAliasIfNotAlreadyExist(Criteria criteria, String associationPath, String newAlias) {
        CriteriaImpl criteriaImpl = (CriteriaImpl) criteria;

        @SuppressWarnings("unchecked")
        Iterator<Subcriteria> it = criteriaImpl.iterateSubcriteria();
        boolean aliasAlreadyExists = false;
        String alias = null;
        while (it.hasNext()) {
            Subcriteria subCriteria = it.next();
            if (associationPath.equals(subCriteria.getPath())) {// identifie la jointure sur l'objet
                aliasAlreadyExists = true;
                alias = subCriteria.getAlias();// l'alias
                //System.out.println("associationPath : '" + associationPath+"' already exist with alias = '" + alias+"'");
                break;
            }
        }
        if (!aliasAlreadyExists) {
            //System.out.println("associationPath : '"+associationPath+"' doesn't exists ");
            criteria.createAlias(associationPath, newAlias);
            alias = newAlias;
            //System.out.println("new one is created");
        }
        return alias;
    }

    /**
     * Returns a query-ready hibernate criteria. Includes pagination, sorting
     * and restrictions.
     *
     * @param pageRequest
     * @return
     */
    public Criteria createCriteria(PageRequest pageRequest) {
        return this.createCriteria(pageRequest, criteria());
    }
}
