/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.messaging.rabbitmq.client;

import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Envelope;
import java.nio.charset.StandardCharsets;
import net.kut3.messaging.Message;
import net.kut3.messaging.rabbitmq.Component;

/**
 *
 */
public class RabbitMQMessage implements Message, Component {

    private final byte[] body;
    private final String bodyAsString;
    
    private String exchangeName = null;
    private String routingKey = null;

    private boolean isRedelivered = false;

    /**
     *
     * @param envelope Message envelope
     * @param props Message properties
     * @param body Message body
     */
    RabbitMQMessage(Envelope envelope, AMQP.BasicProperties props, byte[] body) {

        this.exchangeName = envelope.getExchange();
        this.routingKey = envelope.getRoutingKey();
        this.isRedelivered = envelope.isRedeliver();
        this.body = body;
        this.bodyAsString = new String(body, StandardCharsets.UTF_8);
    }

    /**
     *
     * @param body Message body
     */
    public RabbitMQMessage(String body) {
        this.bodyAsString = body;
        this.body = body.getBytes(StandardCharsets.UTF_8);
    }

    @Override
    public String property(String name) {
        switch (name) {
            case EXCHANGE_NAME:
                return this.exchangeName;

            case ROUTING_KEY:
                return this.routingKey;

            case IS_REDELIVERY:
                return Boolean.toString(this.isRedelivered);

            default:
                return null;
        }
    }

    @Override
    public String bodyAsString() {
        return this.bodyAsString;
    }

    @Override
    public byte[] body() {
        return this.body;
    }
}
