/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.messaging.rabbitmq.client;

import com.rabbitmq.client.ConnectionFactory;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import net.kut3.messaging.Consumer;
import net.kut3.messaging.Producer;
import net.kut3.messaging.client.ClientFactory;
import net.kut3.messaging.client.ConsumerProperties;
import net.kut3.messaging.client.ProducerProperties;
import net.kut3.messaging.rabbitmq.Component;
import net.kut3.messaging.rabbitmq.ExchangeInfo;
import net.kut3.messaging.rabbitmq.QueueInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public class RabbitMQClientFactory implements ClientFactory, Component {

    private static final Logger LOGGER
            = LoggerFactory.getLogger(RabbitMQClientFactory.class);

    private ConnectionFactory connFactory;

    /**
     *
     * @param uri An AMQP connection string URI
     */
    public RabbitMQClientFactory(String uri) {
        ConnectionFactory tmp = new ConnectionFactory();
        try {
            tmp.setUri(uri);
        } catch (URISyntaxException
                | NoSuchAlgorithmException
                | KeyManagementException ex) {

            throw new RuntimeException(ex);
        }

        this.connFactory = tmp;
    }

    @Override
    public Producer newProducer(ProducerProperties props) {
        Producer newProducer;

        ExchangeInfo exchange = (ExchangeInfo) props.get(EXCHANGE_INFO);
        if (null == exchange) {
            newProducer = new ProducerImpl(props.clientName(),
                    this.connFactory,
                    (String) props.get(ROUTING_KEY));
        } else {
            newProducer = new ProducerImpl(props.clientName(),
                    this.connFactory,
                    exchange,
                    (String) props.get(ROUTING_KEY));
        }

        LOGGER.info("Initialized " + newProducer.toString());

        return newProducer;
    }

    @Override
    public Consumer newConsumer(ConsumerProperties props) {
        Consumer newConsumer;

        ExchangeInfo exchange = (ExchangeInfo) props.get(EXCHANGE_INFO);
        if (null == exchange) {
            newConsumer = new ConsumerImpl(props.clientName(),
                    this.connFactory,
                    (QueueInfo) props.get(QUEUE_INFO),
                    props.isAutoAck());
        } else {
            newConsumer = new ConsumerImpl(props.clientName(),
                    this.connFactory,
                    exchange,
                    (QueueInfo) props.get(QUEUE_INFO),
                    (String) props.get(ROUTING_KEY),
                    props.isAutoAck()
            );
        }

        LOGGER.info("Initialized " + newConsumer.toString());

        return newConsumer;
    }
    
    @Override
    public void close() {
    }
}
