/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.messaging.rabbitmq.client;

import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConnectionFactory;
import java.io.IOException;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import net.kut3.messaging.Message;
import net.kut3.messaging.Producer;
import net.kut3.messaging.ProcessResultCode;
import net.kut3.messaging.ProduceResult;
import net.kut3.messaging.rabbitmq.Component;
import net.kut3.messaging.rabbitmq.ExchangeInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
class ProducerImpl extends Client implements Producer, Component {

    private static Logger LOGGER = LoggerFactory.getLogger(ProducerImpl.class);

    private final ExchangeInfo exchangeInfo;
    private final String routingKey;

    /**
     *
     * @param name Name of this procedure
     * @param connFactory A RabbitMQ connection factory
     * @param routingKey Routing key
     */
    ProducerImpl(String name, ConnectionFactory connFactory, String routingKey) {
        this(name, connFactory, new DirectExchangeInfo(), routingKey);
    }

    /**
     *
     * @param name Name of this procedure
     * @param connFactory A RabbitMQ connection factory
     * @param exchangeInfo Information of the exchange
     * @param routingKey Routing key
     */
    ProducerImpl(String name, ConnectionFactory connFactory,
            ExchangeInfo exchangeInfo, String routingKey) {

        super(name, connFactory);

        if (null == exchangeInfo) {
            throw new IllegalArgumentException("exchangeInfo cannot be null");
        }

        if (null == routingKey) {
            throw new IllegalArgumentException("routingKey cannot be null");
        }

        super.declareExchange(exchangeInfo);

        this.exchangeInfo = exchangeInfo;
        this.routingKey = routingKey;
    }

    /**
     *
     * @param name Name of this procedure
     * @param conn A RabbitMQ connection
     */
    ProducerImpl(String name, Connection conn,
            ExchangeInfo exchangeInfo, String routingKey) {

        super(name, conn);

        if (null == exchangeInfo) {
            throw new IllegalArgumentException("exchangeInfo cannot be null");
        }

        if (null == routingKey) {
            throw new IllegalArgumentException("routingKey cannot be null");
        }

        super.declareExchange(exchangeInfo);
        this.exchangeInfo = exchangeInfo;
        this.routingKey = routingKey;
    }

    @Override
    public String name() {
        return super.getName();
    }

    @Override
    public ProduceResult produce(Message message) {
        ProduceResult ret = new ProduceResult();
        try {
            super.channel().basicPublish(this.exchangeInfo.name(),
                    this.routingKey,
                    null,
                    message.body());

            return ret.code(ProcessResultCode.OK);
        } catch (IOException ioEx) {
            return ret.code(ProcessResultCode.ERR_GENERAL)
                    .errorDesc(ioEx.getMessage());
        }
    }

    @Override
    public void close() {
        super.doClose();
    }

    @Override
    public final String toString() {
        return "{\"producerName\":\"" + super.getName()
                + "\", \"exchange\":" + this.exchangeInfo.toString()
                + ", \"routingKey\":\"" + this.routingKey
                + "\"}";
    }
}
