/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.messaging.rabbitmq.client;

import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Channel;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.DefaultConsumer;
import com.rabbitmq.client.Envelope;
import java.io.IOException;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.TimeUnit;
import net.kut3.messaging.Consumer;
import net.kut3.messaging.rabbitmq.QueueInfo;
import net.kut3.messaging.BatchMessageProcessor;
import net.kut3.messaging.MessageProcessor;
import net.kut3.messaging.ProcessResult;
import net.kut3.messaging.ProcessResultCode;
import net.kut3.messaging.rabbitmq.ExchangeInfo;

/**
 *
 */
class ConsumerImpl extends Client implements Consumer {
    
    private final boolean autoAck;
    private final ExchangeInfo exchangeInfo;
    private final QueueInfo queueInfo;
    private final String routingKey;

    /**
     *
     * @param name Name of this consumer
     * @param connFactory A RabbitMQ connection factory
     * @param exchangeInfo Information of the exchange that messages published
     * @param queueInfo Information of queue to consume
     * @param routingKey Routing key to bind exchange and queue
     * @param autoAck Auto acknowledge value
     */
    ConsumerImpl(String name, ConnectionFactory connFactory,
            ExchangeInfo exchangeInfo,
            QueueInfo queueInfo,
            String routingKey,
            boolean autoAck) {
        
        super(name, connFactory);
        
        if (null == queueInfo) {
            throw new IllegalArgumentException("queueInfo cannot be null");
        }
        
        if (null == exchangeInfo) {
            throw new IllegalArgumentException("exchangeInfo cannot be null");
        }
        
        if (null == routingKey) {
            throw new IllegalArgumentException("routingKey cannot be null");
        }
        
        if (exchangeInfo.isDefaultExchange()) {
            if (!routingKey.equals(queueInfo.name())) {
                throw new IllegalArgumentException("If exchange is default exchange, routingKey cannot differ queue name");
            }
            
            super.declareQueue(queueInfo);
        } else {
            super.declareQueue(queueInfo);
            super.declareExchange(exchangeInfo);
            super.bind(exchangeInfo.name(), queueInfo.name(), routingKey);
        }
        
        this.queueInfo = queueInfo;
        this.exchangeInfo = exchangeInfo;
        this.routingKey = routingKey;
        this.autoAck = autoAck;
    }

    /**
     *
     * @param name Name of this consumer
     * @param connFactory A RabbitMQ connection factory
     * @param queueInfo Information of queue to consume
     * @param autoAck
     */
    ConsumerImpl(String name, ConnectionFactory connFactory,
            QueueInfo queueInfo,
            boolean autoAck) {
        
        this(name, connFactory,
                new DirectExchangeInfo(), queueInfo, queueInfo.name(),
                autoAck);
    }
    
    @Override
    public boolean isAutoAck() {
        return this.autoAck;
    }
    
    @Override
    public void start(MessageProcessor processor) {
        try {
            final Channel channel = super.channel();
            channel.basicConsume(this.queueInfo.name(),
                    this.autoAck,
                    new DefaultConsumer(channel) {
                
                @Override
                public void handleDelivery(String consumerTag,
                        Envelope envelope,
                        AMQP.BasicProperties properties,
                        byte[] body) throws IOException {
                    
                    ProcessResult ret
                            = processor.process(new RabbitMQMessage(envelope,
                                    properties, body));
                    
                    if (!autoAck && ret.doAck()) {
                        channel.basicAck(envelope.getDeliveryTag(), false);
                    }
                }
            }
            );
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }
    
    @Override
    public void start(BatchMessageProcessor processor) {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    
    @Override
    public void close() {
        super.doClose();
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{\"consumerName\":\"")
                .append(super.getName())
                .append("\", \"exchange\":").append(this.exchangeInfo.toString())
                .append("\", \"queue\":").append(this.queueInfo.toString())
                .append("\", \"routingKey\":\"").append(this.routingKey)
                .append("\", \"isAutoAck\":").append(this.autoAck);
        
        return sb.append("}").toString();
    }
}
