/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.messaging.rabbitmq.client;

import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.ShutdownSignalException;
import java.io.IOException;
import java.util.concurrent.TimeoutException;
import net.kut3.messaging.rabbitmq.Component;
import net.kut3.messaging.rabbitmq.ExchangeInfo;
import net.kut3.messaging.rabbitmq.QueueInfo;
import org.slf4j.LoggerFactory;

/**
 *
 */
abstract class Client implements Component {

    private final String name;
    private final ConnectionFactory connFactory;
    private final boolean isConnectionOwner;

    private Connection conn;
    private Channel channel;

    /**
     *
     * @param name Name of this producer
     * @param connFactory Connection factory instance
     */
    Client(String name, ConnectionFactory connFactory) {
        if (null == name) {
            throw new IllegalArgumentException("name cannot be null");
        }

        if (null == connFactory) {
            throw new IllegalArgumentException("connFactory cannot be null");
        }

        this.name = name;
        this.connFactory = connFactory;

        try {
            this.conn = this.connFactory.newConnection();
            this.isConnectionOwner = true;
            this.channel = this.conn.createChannel();
        } catch (IOException | TimeoutException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     *
     * @param name Name of this client
     * @param conn Provided connection
     */
    Client(String name, Connection conn) {
        if (null == name) {
            throw new IllegalArgumentException("name cannot be null");
        }

        if (null == conn) {
            throw new IllegalArgumentException("conn cannot be null");
        }

        this.isConnectionOwner = false;
        this.name = name;
        this.conn = conn;
        this.connFactory = null;
    }

    /**
     *
     * @param ioEx An {@link IOException} to extract shutdown cause
     * @return Shutdown cause
     */
    public static Object tryToExtractShutdownCause(IOException ioEx) {
        if (null != ioEx.getCause()) {
            ShutdownSignalException innerEx
                    = (ShutdownSignalException) ioEx.getCause();

            if (null != innerEx) {
                if (!innerEx.isHardError()) {
                    switch (innerEx.getReason().protocolMethodId()) {
                        case 40:
                            com.rabbitmq.client.impl.AMQImpl.Channel.Close channelCloseCause
                                    = (com.rabbitmq.client.impl.AMQImpl.Channel.Close) innerEx.getReason();

                            switch (channelCloseCause.getReplyCode()) {
                                case 406:
                                    return channelCloseCause.getReplyText();

                                default:
                                    break;
                            }

                            break;

                        default:
                            break;
                    }
                }
            }
        }

        return null;
    }

    /**
     *
     * @return Name of this client
     */
    protected final String getName() {
        return this.name;
    }

    /**
     *
     * @return The current {@link Channel} object
     */
    protected final Channel channel() {
        return this.channel;
    }

    /**
     * @param queueInfo Queue information
     */
    protected final void declareQueue(QueueInfo queueInfo) {
        try (Channel tmpChannel = this.conn.createChannel()) {
            tmpChannel.queueDeclare(queueInfo.name(),
                    queueInfo.isDurable(),
                    queueInfo.isTemporary(),
                    queueInfo.isTemporary(),
                    null);

            LoggerFactory.getLogger(this.getClass())
                    .info("'" + this.name + "' declared queue "
                            + queueInfo.toString() + " successfully");
        } catch (IOException ex) {
            Object o = tryToExtractShutdownCause(ex);
            if (null != o) {
                throw new IllegalArgumentException(o.toString());
            }

            throw new RuntimeException(ex);
        } catch (TimeoutException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     *
     * @param exchangeInfo Information of the exchange to be declared
     */
    protected final void declareExchange(ExchangeInfo exchangeInfo) {
        if ("".equals(exchangeInfo.name())) {
            return;
        }

        try (Channel tmpChannel = this.conn.createChannel()) {
            tmpChannel.exchangeDeclare(exchangeInfo.name(),
                    exchangeInfo.bindingType(),
                    exchangeInfo.isDurable(),
                    false,
                    null);

            LoggerFactory.getLogger(this.getClass())
                    .info("'" + this.name + "' declared exchange "
                            + exchangeInfo.toString() + " successfully");
        } catch (IOException ex) {
            Object o = tryToExtractShutdownCause(ex);
            if (null != o) {
                throw new IllegalArgumentException(o.toString());
            }

            throw new RuntimeException(ex);
        } catch (TimeoutException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     *
     * @param exchangeName Name of the exchange to bind
     * @param queueName Name of the queue to bind
     * @param routingKey Routing key from exchange to queue
     */
    protected final void bind(String exchangeName, String queueName,
            String routingKey) {

        if (null == exchangeName) {
            throw new IllegalArgumentException("exchangeName cannot be null");
        }

        if ("".equals(exchangeName)) {
            throw new IllegalArgumentException("Default exchange cannot bind any queues explicitly");
        }

        if (null == queueName) {
            throw new IllegalArgumentException("queueName cannot be null");
        }

        try (Channel tmpChannel = this.conn.createChannel()) {
            tmpChannel.queueBind(queueName, exchangeName, routingKey);
        } catch (IOException | TimeoutException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     *
     */
    protected final void doClose() {
        if (null != this.channel) {
            try {
                this.channel.close();
                LoggerFactory.getLogger(this.getClass()).info("'" + this.name
                        + "' closed current channel successfully");
            } catch (TimeoutException | IOException ex) {
                LoggerFactory.getLogger(this.getClass())
                        .error("'" + this.name + "' closed channel error", ex);
            }

            this.channel = null;
        }

        if (this.isConnectionOwner && null != this.conn) {
            try {
                this.conn.close();
                LoggerFactory.getLogger(this.getClass()).info("'" + this.name
                        + "' closed current connection successfully");
            } catch (IOException ex) {
                LoggerFactory.getLogger(this.getClass())
                        .error("'" + this.name + "' closed connection error", ex);
            }

            this.conn = null;
        }
    }
}
