/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.messaging.kafka.client;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import net.kut3.messaging.client.ConsumerProperties;
import net.kut3.messaging.kafka.Component;
import static net.kut3.messaging.kafka.Component.*;
import net.kut3.messaging.kafka.OffsetResetMode;
import static org.apache.kafka.clients.consumer.ConsumerConfig.*;
import org.apache.kafka.common.serialization.StringDeserializer;

/**
 *
 */
public class SimpleConsumerProperties extends ConsumerProperties
        implements Component {

    /**
     *
     * @param producerName Name of producer
     * @param servers Bootstrap servers addresses
     * @param groupId Id of the group this consumer in
     * @param topics Topics to consume
     */
    public SimpleConsumerProperties(String producerName, String servers,
            String groupId, Collection<String> topics) {

        this(producerName, servers, groupId, null, topics);
    }

    /**
     *
     * @param consumerName Name of consumer
     * @param servers Bootstrap servers addresses
     * @param groupId Id of the group this consumer in
     * @param offsetResetMode Offset reset mode
     * @param topics Topics to consume
     */
    public SimpleConsumerProperties(String consumerName, String servers,
            String groupId, OffsetResetMode offsetResetMode,
            Collection<String> topics) {

        super(consumerName);

        if (null == servers || servers.trim().length() == 0) {
            throw new IllegalArgumentException("servers cannot be null or left blank");
        }

        if (null == topics || topics.isEmpty()) {
            throw new IllegalArgumentException("topics cannot be null or empty");
        }

        super.put(TOPICS, topics);

        Map<String, Object> props = new HashMap<>();
        props.put(BOOTSTRAP_SERVERS_CONFIG, servers);
        props.put(CLIENT_ID_CONFIG, consumerName);
        props.put(GROUP_ID_CONFIG, groupId);
        props.put(KEY_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);
        props.put(VALUE_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);

        if (null != offsetResetMode) {
            props.put(AUTO_OFFSET_RESET_CONFIG, offsetResetMode.asString());
        }

        super.put(CONSUMER_PROPERTIES, props);
    }
}
