/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.messaging.kafka;

import java.util.HashMap;
import java.util.Map;
import org.apache.kafka.clients.producer.ProducerRecord;
import static org.apache.kafka.clients.producer.ProducerConfig.*;

/**
 *
 */
public final class KafkaProducer {

    private final ProducerBuilder builder;

    private org.apache.kafka.clients.producer.KafkaProducer<String, String> producer;

    private KafkaProducer(ProducerBuilder builder) {
        this.builder = builder;

        this.initProducer();
    }

    /**
     *
     * @param key Key to produce
     * @param value Value to produce
     */
    public void produce(String key, String value) {
        this.producer.send(new ProducerRecord<>(this.builder.topic,
                key, value)
        );
    }

    /**
     *
     */
    public void close() {
        this.producer.close();
    }

    private void initProducer() {
        this.producer = new org.apache.kafka.clients.producer.KafkaProducer<>(this.builder.props);
    }

    /**
     *
     * @param servers One or more host port pairs separated by commas.
     * @param topic The topic to send messages.
     * @param clientId The id of this client.
     * @return new {@link ProducerBuilder}
     */
    public static ProducerBuilder newBuilder(String servers, String topic,
            String clientId) {

        return new ProducerBuilder(servers, topic, clientId);
    }

    /**
     *
     */
    public static final class ProducerBuilder {

        private final Map<String, Object> props = new HashMap<>();
        private final String topic;

        private ProducerBuilder(String servers, String topic, String clientId) {
            this.topic = topic;
            this.props.put(BOOTSTRAP_SERVERS_CONFIG, servers);
            this.props.put(CLIENT_ID_CONFIG, clientId);

            this.props.put(ACKS_CONFIG, "all");
            this.props.put(DELIVERY_TIMEOUT_MS_CONFIG, 31000);
            this.props.put(BATCH_SIZE_CONFIG, 16384);
            this.props.put(LINGER_MS_CONFIG, 1);
            this.props.put(BUFFER_MEMORY_CONFIG, 33554432);
            this.props.put(KEY_SERIALIZER_CLASS_CONFIG,
                    "org.apache.kafka.common.serialization.StringSerializer");

            this.props.put(VALUE_SERIALIZER_CLASS_CONFIG,
                    "org.apache.kafka.common.serialization.StringSerializer");
        }

        /**
         *
         * @return A new {@link KafkaProducer} instance
         */
        public KafkaProducer build() {
            return new KafkaProducer(this);
        }
    }

    /**
     *
     * @param args Input parameters.
     */
    public static void main(String[] args) {

        KafkaProducer producer = KafkaProducer
                .newBuilder("10.1.1.99:9092,10.1.1.99:9093,10.1.1.98:9094",
                        "my-topic",
                        "producer-001")
                .build();

        producer.produce("key", "value");

        producer.close();
    }
}
