/*
 * Copyright 2019 Kut3Net.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.kut3.data.elt;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import net.kut3.data.DbClient;
import net.kut3.data.mongo.MongoFunction;
import net.kut3.messaging.BatchMessageProcessor;
import net.kut3.messaging.Message;
import net.kut3.messaging.Consumer;
import net.kut3.messaging.ProcessResult;
import org.bson.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public class Kafka2MongoLoader implements ELTProcess {

    private static final Logger LOGGER
            = LoggerFactory.getLogger(Kafka2MongoLoader.class);

    private final Consumer consumer;
    private final DbClient dbClient;
    private final String targetDb;
    private final String targetCollection;

    /**
     *
     * @param consumer A Kafka consumer
     * @param mongoClient A Mongo client
     * @param targetDb Name of the Mongo database to insert
     * @param targetCollection Name of the Mongo collection to insert
     */
    public Kafka2MongoLoader(Consumer consumer,
            DbClient mongoClient, String targetDb, String targetCollection) {

        this.targetDb = targetDb;
        this.targetCollection = targetCollection;

        this.consumer = consumer;
        this.dbClient = mongoClient;

        LOGGER.info(this.toString());
    }

    @Override
    public void process() {
        try (Connection conn = this.dbClient.getConnection()) {
            final PreparedStatement ps = conn.prepareStatement(MongoFunction.INSERT_MANY.value());
            ps.setString(1, this.targetDb);
            ps.setString(2, this.targetCollection);
            ps.setBoolean(4, true);

            final List<Map<String, Object>> docs = new ArrayList<>();

            this.consumer.start(new BatchMessageProcessor() {
                @Override
                public void addToBatch(Message message) {
                    LOGGER.info("addToBatch(): " + message.bodyAsString());

                    Document tmp = Document.parse(message.bodyAsString());
                    replaceKey(tmp);

                    docs.add(tmp);
                }

                @Override
                public List<ProcessResult> processBatch() {
                    try {
                        ps.setObject(3, docs);
                        ps.executeUpdate();
                    } catch (SQLException sqlEx) {
                        LOGGER.error("", sqlEx);
                    }

                    docs.clear();
                    return new ArrayList<>(0);
                }
            });
        } catch (SQLException | RuntimeException | Error ex) {
            ex.printStackTrace(System.err);
        }
    }

    @Override
    public void cancel() {
        try {
            this.consumer.close();
        } catch (RuntimeException | Error ex) {
            LOGGER.warn(this.consumer.name() + ".close() error", ex);
        }
        
        try {
            this.dbClient.close();
        } catch (RuntimeException | Error ex) {
            LOGGER.warn(this.dbClient.toString() + ".close() error", ex);
        }
    }

    private void replaceKey(Document doc) {
        new HashSet<>(doc.keySet()).forEach(k -> {
            Object o;
            if (k.contains(".")) {
                o = doc.remove(k);
                doc.put(k.replace(".", "-"), o);
            } else {
                o = doc.get(k);
            }

            if (o instanceof Document) {
                replaceKey((Document) o);
            }
        });
    }

    @Override
    public final String toString() {
        return "{consumer=" + this.consumer.toString()
                + ", dbClient=" + this.dbClient.toString()
                + ", targetDb=" + this.targetDb
                + ", targetCollection=" + this.targetCollection
                + "}";
    }
}
