package net.kigawa.renlin.dsl

import net.kigawa.hakate.api.state.State
import net.kigawa.renlin.AutoFill
import net.kigawa.renlin.component.*
import net.kigawa.renlin.state.DslState
import net.kigawa.renlin.tag.Fragment
import net.kigawa.renlin.tag.Tag
import net.kigawa.renlin.w3c.category.ContentCategory
import net.kigawa.renlin.w3c.element.TagNode

/**
 * `Dsl` インターフェースは、HTML構造を構築するためのDSL（ドメイン特化言語）の基本機能を定義します。
 * このインターフェースは、コンテンツカテゴリに基づいたHTML要素の構築と状態管理を提供します。
 *
 * 主な機能:
 * - HTML要素の構築と管理
 * - コンポーネントの描画と状態管理
 * - サブDSLの管理とマウント
 * - 状態値の使用と追跡
 *
 * 使用場所:
 * - HTML構造を構築するDSLの実装
 * - コンポーネントベースのUI構築
 * - 状態管理を伴うHTML生成
 *
 * @param CONTENT_CATEGORY このDSLが生成できるHTMLコンテンツのカテゴリ
 */
interface StatedDsl<CONTENT_CATEGORY: ContentCategory>: Dsl {
    /**
     * 現在のDSLに関連付けられた状態。
     * この状態はDSLの動作と描画を制御します。
     */
    val dslState: DslState

    /**
     * サブDSLを現在のDSLに登録します。
     *
     * @param registeredDslData 登録するDSLのデータ
     */
    fun registerSubDsl(registeredDslData: RegisteredDslData)

    /**
     * 指定された状態をDSLにマウントします。
     * これにより、DSLの状態が更新され、関連するサブDSLも更新されます。
     *
     * @param state マウントする状態
     * @param registeredDslData 関連するDSLデータ
     */
    fun applyToDslState(state: DslState, registeredDslData: RegisteredDslData)

    /**
     * 指定された要素をDSLに適用します。
     *
     * @param element 適用するHTML要素
     * @return 要素の適用を取り消すための関数
     */
    fun applyElement(element: TagNode): () -> Unit


    operator fun <TAG: Tag<in CONTENT_CATEGORY>> Component0<TAG, in CONTENT_CATEGORY>.invoke(
        @AutoFill key: String? = null,
    ) = this.render(this@StatedDsl, key)

    /**
     * コンポーネントを呼び出し、レンダリングするための演算子オーバーロード。
     *
     * @param key コンポーネントの一意のキー（オプション）
     * @param block コンポーネントの内容を定義するブロック
     * @return コンポーネントのレンダリング結果
     */
    operator fun <TAG: Tag<in CONTENT_CATEGORY>, DSL> Component1<TAG, in CONTENT_CATEGORY, DSL>.invoke(
        @AutoFill key: String? = null, block: DSL,
    ) {
        return this@invoke.render(this@StatedDsl, block, key)
    }

    /**
     * Component2を呼び出し、レンダリングするための演算子オーバーロード。
     *
     * @param key コンポーネントの一意のキー（オプション）
     * @param arg1 第1引数
     * @param arg2 第2引数
     * @return コンポーネントのレンダリング結果
     */
    operator fun <TAG: Tag<in CONTENT_CATEGORY>, ARG1, ARG2> Component2<TAG, in CONTENT_CATEGORY, ARG1, ARG2>.invoke(
        arg1: ARG1, arg2: ARG2, @AutoFill key: String? = null,
    ) {
        return this@invoke.render(this@StatedDsl, arg1, arg2, key)
    }

    /**
     * Component3を呼び出し、レンダリングするための演算子オーバーロード。
     *
     * @param key コンポーネントの一意のキー（オプション）
     * @param arg1 第1引数
     * @param arg2 第2引数
     * @param arg3 第3引数
     * @return コンポーネントのレンダリング結果
     */
    operator fun <TAG: Tag<in CONTENT_CATEGORY>, ARG1, ARG2, ARG3> Component3<TAG, in CONTENT_CATEGORY, ARG1, ARG2, ARG3>.invoke(
        arg1: ARG1, arg2: ARG2, arg3: ARG3, @AutoFill key: String? = null,
    ) {
        return this@invoke.render(this@StatedDsl, arg1, arg2, arg3, key)
    }

    /**
     * Component4を呼び出し、レンダリングするための演算子オーバーロード。
     *
     * @param key コンポーネントの一意のキー（オプション）
     * @param arg1 第1引数
     * @param arg2 第2引数
     * @param arg3 第3引数
     * @param arg4 第4引数
     * @return コンポーネントのレンダリング結果
     */
    operator fun <TAG: Tag<in CONTENT_CATEGORY>, ARG1, ARG2, ARG3, ARG4> Component4<TAG, in CONTENT_CATEGORY, ARG1, ARG2, ARG3, ARG4>.invoke(
        arg1: ARG1, arg2: ARG2, arg3: ARG3, arg4: ARG4, @AutoFill key: String? = null,
    ) {
        return this@invoke.render(this@StatedDsl, arg1, arg2, arg3, arg4, key)
    }


    /**
     * フラグメントコンポーネント。
     * これは、複数の要素をグループ化するためのコンテナとして機能します。
     */
    val fragment: StructuredComponent<Tag<CONTENT_CATEGORY>, CONTENT_CATEGORY, out StatedDsl<CONTENT_CATEGORY>>
        get() = Fragment.create<CONTENT_CATEGORY>()

    /**
     * 状態の現在の値を取得し、この状態をDSLの状態セットに追加します。
     *
     * @return 状態の現在の値
     */
    fun <T> State<T>.useValue(): T
}
