package net.kensand.kielbasa.coroutines

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.flow.flow

/**
 * Repeatedly, indefinitely, and lazily call the function and emit the result to a flow.
 * @param A function that returns type T.
 * @return A Flow of type T.
 */
fun <T> repeatAsFlow(block: () -> T) =
    flow {
        while (true) {
            emit(block())
        }
    }

/**
 * Repeatedly, indefinitely, and lazily call the receiver function and emit the result to a flow, while holding the result in state for the next iteration.
 * @param initial The initial state passed into the receiver function.
 * @param skipInitial Skip emission of the initial value. False by default.
 * @param block The function that will be called indefinitely using the return value from the previous iteration.
 * @return A Flow of results from the receiver function, optionally starting with the given initial value.
 */
fun <T> repeatAsFlow(
    initial: T,
    skipInitial: Boolean = false,
    block: ((T) -> T),
) = flow {
    var state: T = initial
    if (!skipInitial) {
        emit(state)
    }
    while (true) {
        state = block(state)
        emit(state)
    }
}

/**
 * Coroutine-safe runCatching.
 * Calls the provided block and encapsulates the result or any throwable in a Result.
 * If the throwable is a CancellationException, it is re-thrown.
 *
 * @param T The expected type of successful calls to block.
 * @param block The block function for which exceptions should be caught.
 */
suspend fun <T> coRunCatching(block: suspend () -> T) =
    try {
        Result.success(block())
    } catch (e: CancellationException) {
        throw e
    } catch (e: kotlin.coroutines.cancellation.CancellationException) {
        throw e
    } catch (e: Throwable) {
        Result.failure(e)
    }

/**
 * Coroutine-safe runCatching.
 * Calls the provided block and encapsulates the result or any throwable in a Result.
 * If the throwable is a CancellationException, it is re-thrown.
 *
 * @param T The expected type of successful calls to block.
 * @param block The block function for which exceptions should be caught.
 * @receiver The scope within which to run the block.
 */
suspend fun <R, T> R.coRunCatching(block: suspend R.() -> T) =
    net.kensand.kielbasa.coroutines
        .coRunCatching { this.block() }
