package net.kensand.kielbasa.coroutines

import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.asFlow
import kotlinx.coroutines.flow.flow

/**
 * Break a flow into chunks, similar to [Collection.chunked].
 * The last chunk may be smaller than chunkSize.
 *
 * @param chunkSize The max size of each chunk.
 * @return A flow of chunks of elements, each of which is also a flow.
 */
suspend fun <T> Flow<T>.chunked(chunkSize: Int): Flow<Flow<T>> {
    require(chunkSize > 0) {
        "chunkSize must be greater than zero"
    }
    return flow {
        var buffer = emptyList<T>()
        this@chunked.collect {
            buffer = buffer + listOf(it)
            if (buffer.size >= chunkSize) {
                emit(buffer.asFlow())
                buffer = emptyList()
            }
        }
        if (buffer.isNotEmpty()) {
            emit(buffer.asFlow())
        }
    }
}
