package net.kensand.kielbasa.core

import kotlin.js.ExperimentalJsExport
import kotlin.js.JsExport

/**
 * Convert a Collection into a Map keyed by the Collections indices.
 * @receiver The collection to be indexed and converted to a map.
 * @return A mapping from indices to their elements in the receiver collection.
 */
fun <E> Collection<E>.toMap() = withIndex().associate { it.index to it.value }

/**
 * Get the element following the first element matched by the predicate.
 * @param distance The distance of the element following the first match that should be returned. Defaults to 1.
 * @param predicate The predicate used to match the element. Defaults to { true }.
 * @return The element that is the given distance after the first predicate match, or null if no match is found.
 */
@ExperimentalJsExport
@JsExport
fun <E> Collection<E>.afterFirstOrNull(
    distance: Int = 1,
    predicate: (E) -> Boolean = { true },
): E? =
    toList().run {
        indexOfFirst(predicate).let { index ->
            if (index == -1) {
                null
            } else {
                this.getOrNull(index + distance)
            }
        }
    }

/**
 * Get the element following the last element matched by the predicate.
 * @param distance The distance of the element following the last match that should be returned. Negative distances are valid. Defaults to 1.
 * @param predicate The predicate used to match the element. Defaults to { true }.
 * @return The element that is the given distance after the last predicate match, or null if no match is found.
 */
@ExperimentalJsExport
@JsExport
fun <E> Collection<E>.afterLastOrNull(
    distance: Int = 1,
    predicate: (E) -> Boolean = { true },
): E? =
    toList().run {
        indexOfLast(predicate).let { index ->
            if (index == -1) {
                null
            } else {
                this.getOrNull(index + distance)
            }
        }
    }

/**
 * Get the element following the last element matched by the predicate.
 * @param distance The distance of the element following the last match that should be returned. Negative distances are valid. Defaults to 1.
 * @param predicate The predicate used to match the element. Defaults to { true }.
 * @return The element that is the given distance after the last predicate match.
 * @throws IndexOutOfBoundsException When the desired element is out of the Receiver's bounds.
 */
@ExperimentalJsExport
@JsExport
fun <E> Collection<E>.afterLast(
    distance: Int = 1,
    predicate: (E) -> Boolean = { true },
): E =
    toList().run {
        indexOfLast(predicate).let { index ->
            if (index == -1) {
                throw IndexOutOfBoundsException("No element matched the given predicate.")
            } else {
                this.getOrNull(index + distance)
                    ?: throw IndexOutOfBoundsException("Index ${index + distance} is out of bounds for Collection of size $size")
            }
        }
    }

/**
 * Get the element following the first element matched by the predicate.
 * @param distance The distance of the element following the first match that should be returned. Negative distances are valid. Defaults to 1.
 * @param predicate The predicate used to match the element. Defaults to { true }.
 * @return The element that is the given distance after the first predicate match.
 * @throws IndexOutOfBoundsException When the desired element is out of the Receiver's bounds.
 */
@ExperimentalJsExport
@JsExport
fun <E> Collection<E>.afterFirst(
    distance: Int = 1,
    predicate: (E) -> Boolean = { true },
): E =
    toList().run {
        indexOfFirst(predicate).let { index ->
            if (index == -1) {
                throw IndexOutOfBoundsException("No element matched the given predicate.")
            } else {
                this.getOrNull(index + distance)
                    ?: throw IndexOutOfBoundsException("Index ${index + distance} is out of bounds for Collection of size $size")
            }
        }
    }

/**
 * Drop the first elements while a predicate matches.
 *
 * @param predicate The predicate determining if the first element should be dropped,
 * @return A new list with the beginning elements for which the predicate returns true removed.
 */
fun <E> Collection<E>.dropFirstWhile(predicate: (E) -> Boolean): List<E> =
    toList().run {
        indexOfFirst { !predicate(it) }.let {
            if (it == -1) {
                emptyList()
            } else {
                subList(it, size)
            }
        }
    }

/**
 * Drop the last elements while a predicate matches.
 *
 * @param predicate The predicate determining if the last element should be dropped,
 * @return A new list with the end elements for which the predicate returns true removed.
 */
fun <E> Collection<E>.dropLastWhile(predicate: (E) -> Boolean): List<E> = toList().reversed().dropFirstWhile(predicate).reversed()

/**
 * Index of first or null
 *
 * @param predicate The predicate used to match elements against.
 * @return The index of the first element for which the [predicate] returns true, or null if the [predicate] returns false for all elements.
 */
@ExperimentalJsExport
@JsExport
fun <E> Collection<E>.indexOfFirstOrNull(predicate: (E) -> Boolean): Int? = indexOfFirst(predicate).takeIf { it != -1 }

/**
 * Index of last or null
 *
 * @param predicate The predicate used to match elements against.
 * @return The index of the last element for which the [predicate] returns true, or null if the [predicate] returns false for all elements.
 */
@ExperimentalJsExport
@JsExport
fun <E> Collection<E>.indexOfLastOrNull(predicate: (E) -> Boolean): Int? = indexOfLast(predicate).takeIf { it != -1 }
