/**
 * The MIT License (MIT)
 *
 * Copyright (c) 2017 Paul Campbell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.kemitix.quality.api;

import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.MojoExecutionException;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import java.util.List;
import java.util.Objects;

/**
 * A Plugin and a Goal of that plugin.
 *
 * @author Paul Campbell (pcampbell@kemitix.net)
 */
public interface PluginGoal {

    /**
     * Get the Plugin.
     *
     * @return the Plugin
     */
    Plugin getPlugin();

    /**
     * Get the Goals for the Plugin.
     *
     * @return the goal names
     */
    List<String> getGoals();

    /**
     * Set the maven coordinates for the plugin.
     *
     * @param groupId    The groupId
     * @param artifactId The artifactId
     * @param version    The version
     */
    void groupArtifactVersion(String groupId, String artifactId, String version);

    /**
     * Configure the Plugin Goal.
     *
     * @param configuration The configuration
     */
    void configuration(PluginGoalConfiguration configuration);

    /**
     * Gets the Configuration of the plugin.
     *
     * @return the Xpp3Dom configuration
     */
    Xpp3Dom getConfiguration();

    /**
     * Extract the value of a configuration element by name.
     *
     * @param elementName the name of the element
     *
     * @return the value of the element
     */
    default String configValue(final String elementName) {
        final Plugin plugin = getPlugin();
        final Xpp3Dom configuration = (Xpp3Dom) plugin.getConfiguration();
        final Xpp3Dom element = configuration.getChild(elementName);
        Objects.requireNonNull(element, "missing element: " + elementName);
        return element.getValue();
    }

    /**
     * Hook called just before the plugin is executed.
     *
     * <p>This default implementation does nothing.</p>
     *
     * @param goal The goal
     */
    default void onExecute(final String goal) {
        // does nothing
    }

    /**
     * Hook called just after the plugin is executed without throwing any exceptions.
     *
     * <p>This default implementation does nothing.</p>
     *
     * @param goal The goal
     */
    default void onSuccess(final String goal) {
        // does nothing
    }

    /**
     * Hook called if there is an error when the plugin is executed.
     *
     * <p>This default implementation does nothing.</p>
     *
     * @param goal The goal
     * @param e    The exception that was thrown
     */
    default void onError(final String goal, final MojoExecutionException e) {
        // does nothing
    }

    /**
     * Hook called just after the plugin is executed, whether there is any error or not.
     *
     * <p>This default implementation does nothing.</p>
     *
     * @param goal The goal
     */
    default void onComplete(final String goal) {
        // does nothing
    }

    /**
     * Returns the name of the plugin-goal.
     *
     * <p>e.g. This is the name used when a plugin needs to be disabled.</p>
     *
     * @return the name of the plugin goal
     */
    String getName();

    /**
     * Defines the weight of the plugin goal.
     *
     * <p>Lighter (lower numberically) plugin goals are executed first.</p>
     *
     * @return the weight of the plugin goal
     */
    default Integer getWeight() {
        return 0;
    }

    /**
     * Defines the maven lifecycle phase to run the plugin in.
     *
     * @param lifecyclePhase the phase
     *
     * @return true if the phase is the same
     */
    default boolean matches(final LifecyclePhase lifecyclePhase) {
        return LifecyclePhase.VERIFY.equals(lifecyclePhase);
    }

    /**
     * Maven Lifecycles to run PluginGoals in.
     */
    enum LifecyclePhase {
        VALIDATE,
        VERIFY
    }
}
