joo.classLoader.prepare(////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2013 CodeCatalyst, LLC - http://www.codecatalyst.com/
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.	
////////////////////////////////////////////////////////////////////////////////

"package com.codecatalyst.promise",/*
{
	import com.codecatalyst.promise.logger.LogLevel;
	import com.codecatalyst.util.nextTick;
	import com.codecatalyst.util.optionally;
	import com.codecatalyst.util.spread;
	
	import flash.events.TimerEvent;
	import flash.utils.Timer;*/

	/**
	 * Promises represent a future value; i.e., a value that may not yet be available.
	 */
	"public class Promise",1,function($$private){var as=joo.as,is=joo.is,$$bound=joo.boundMethod,$3=com.codecatalyst.promise,$4=flash.events,$5=flash.utils,$6=com.codecatalyst.util,$7=com.codecatalyst.promise.logger;return[function(){joo.classLoader.init(com.codecatalyst.promise.logger.LogLevel,flash.events.TimerEvent);},
	
		// ========================================
		// Public static methods
		// ========================================

		/**
		 * Returns a new Promise of the specified value, which may be an
		 * immediate value, a Promise, or a foreign Promise (i.e. Promises 
		 * from another Promises/A implementation).
		 * 
		 * Additionally, the specified value can be adapted into a Promise
		 * through the use of custom adapters.
		 * 
		 * @param value An immediate value, a Promise, a foreign Promise or adaptable value.
		 * @return Promise of the specified value.
		 * 
		 * @see #registerAdapter()
		 * @see #unregisterAdapter()
		 * @see com.codecatalyst.promise.adapter.AsyncTokenAdapter
		 */
		"public static function when",function when( value/*:**/ )/*:Promise*/
		{
			for/* each*/ (var $1=0;$1</* in*/ $$private.adapters.length;++$1 )
			{ var adapt/*:Function*/ = $$private.adapters[$1];/*
				const*/var promise/*:Promise*/ =as( adapt( value ),  $3.Promise);
				if ( promise )
				{
					return promise;
				}
			}/*
			
			const*/var deferred/*:Deferred*/ = new $3.Deferred();
			deferred.resolve( value );
			return deferred.promise;
		},
		
		/**
		 * Determines whether the specified value is a thenable, i.e. an Object
		 * or Function that exposes a then() method and may therefore be an 
		 * third-party untrusted Promise, based on the Promises/A 
		 * specification feature test.
		 * 
		 * @param value A potential thenable.
		 * @return Boolean indicating whether the specified value was a thenable.
		 */
		"public static function isThenable",function isThenable( value/*:**/ )/*:Boolean*/
		{
			return ( value != null && (is( value,  Object) ||is( value,  Function) ) && "then" in value &&is( value.then,  Function) );
		},
		
		/**
		 * Returns a new Promise that will only fulfill once all the specified
		 * Promises and/or values have been fulfilled, and will reject if any
		 * of the specified Promises is rejected. The resolution value will be 
		 * an Array containing the fulfilled value of each of the Promises or 
		 * values.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @returns Promise of an Array of the fulfilled values.
		 */
		"public static function all",function all( promisesOrValues/*:**/ )/*:Promise*/
		{
			if ( ! (is( promisesOrValues,  Array) || $3.Promise.isThenable( promisesOrValues ) ) )
			{
				throw new Error( "Invalid parameter: expected an Array or Promise of an Array." );
			}
			
			function process( promisesOrValues/*:Array*/ )/*:Promise*/
			{
				var remainingToResolve/*:uint*/ = promisesOrValues.length;
				var results/*:Array*/ = new Array( promisesOrValues.length );
				
				var deferred/*:Deferred*/ = new $3.Deferred();
				
				if ( remainingToResolve > 0 )
				{
					function resolve( item/*:**/, index/*:uint*/ )/*:Promise*/
					{
						function fulfill( value/*:**/ )/*:**/
						{
							results[ index ] = value;
							if ( --remainingToResolve == 0 )
							{
								deferred.resolve( results );
							}
							return value;
						}
						
						return $3.Promise.when( item ).then( fulfill,$$bound( deferred,"reject") );
					}
					
					for ( var index/*:uint*/ = 0; index < promisesOrValues.length; index++ )
					{
						if ( index in promisesOrValues )
						{
							resolve( promisesOrValues[ index ], index );
						}
						else
						{
							remainingToResolve--;
						}
					}
				}
				else
				{
					deferred.resolve( results );
				}
				
				return deferred.promise;
			}
			
			return $3.Promise.when( promisesOrValues ).then( process );
		},
		
		/**
		 * Initiates a competitive race, returning a new Promise that will 
		 * fulfill when any one of the specified Promises or values is 
		 * fulfilled, or will only reject once all of the Promises have
		 * been rejected.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @return Promise of the first resolved value.
		 */
		"public static function any",function any( promisesOrValues/*:**/ )/*:Promise*/
		{
			if ( ! (is( promisesOrValues,  Array) || $3.Promise.isThenable( promisesOrValues ) ) )
			{
				throw new Error( "Invalid parameter: expected an Array or Promise of an Array." );
			}
			
			function extract( array/*:Array*/ )/*:**/
			{
				return array[ 0 ];
			}
			
			function transform( reason/*:**/ )/*:void*/
			{
				if (is( reason,  Error) && reason.message == "Too few Promises were resolved." )
				{
					throw new Error( "No Promises were resolved." );
				}
				throw reason;
			}

			return $3.Promise.some( promisesOrValues, 1 ).then( extract, transform );
		},
		
		/**
		 * Initiates a competitive race, returning a new Promise that will 
		 * fulfill when the expected number of Promises and/or values have
		 * been fulfilled, or will reject when it becomes impossible for the
		 * expected number to fulfill.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @param howMany The expected number of fulfilled values.
		 * @return Promise of the expected number of fulfilled values.
		 */
		"public static function some",function some( promisesOrValues/*:**/, howMany/*:uint*/ )/*:Promise*/
		{
			if ( ! (is( promisesOrValues,  Array) || $3.Promise.isThenable( promisesOrValues ) ) )
			{
				throw new Error( "Invalid parameter: expected an Array or Promise of an Array." );
			}
			
			function process( promisesOrValues/*:Array*/ )/*:Promise*/
			{
				var values/*:Array*/ = [];
				var remainingToResolve/*:uint*/ = howMany;
				var remainingToReject/*:uint*/ = ( promisesOrValues.length - remainingToResolve ) + 1;
				
				var deferred/*:Deferred*/ = new $3.Deferred();
				
				if ( promisesOrValues.length < howMany )
				{
					deferred.reject( new Error( "Too few Promises were resolved." ) );
				}
				else
				{
					function onResolve( value/*:**/ )/*:**/
					{
						if ( remainingToResolve > 0 ) {
							values.push( value );
						}
						remainingToResolve--;
						if ( remainingToResolve == 0 )
						{
							deferred.resolve( values );
						}
						return value;
					}
					
					function onReject( reason/*:**/ )/*:**/
					{
						remainingToReject--;
						if ( remainingToReject == 0 )
						{
							deferred.reject( new Error( "Too few Promises were resolved." ) );
						}
						throw reason;
					}
					
					for ( var index/*:uint*/ = 0; index < promisesOrValues.length; index++ )
					{
						if ( index in promisesOrValues )
						{
							$3.Promise.when( promisesOrValues[ index ] ).then( onResolve, onReject );
						}
					}
				}
				
				return deferred.promise;
			}
			
			return $3.Promise.when( promisesOrValues ).then( process );
		},
		
		/**
		 * Returns a new Promise that will automatically resolve with the 
		 * specified Promise or value after the specified delay 
		 * (in milliseconds).
		 *
		 * @param promiseOrValue A Promise or value.
		 * @param milliseconds Delay duration (in milliseconds).
		 * @return Promise of the specified Promise or value that will resolve after the specified delay.
		 */
		"public static function delay",function delay( promiseOrValue/*:**/, milliseconds/*:Number*/ )/*:Promise*/
		{
			var deferred/*:Deferred*/ = new $3.Deferred();
			
			function timerCompleteHandler()/*:void*/
			{
				timer.removeEventListener( $4.TimerEvent.TIMER_COMPLETE, timerCompleteHandler );
				
				deferred.resolve( promiseOrValue );
			}
			
			var timer/*:Timer*/ = new $5.Timer( Math.max( milliseconds, 0 ), 1 );
			timer.addEventListener( $4.TimerEvent.TIMER_COMPLETE, timerCompleteHandler );
			
			timer.start(); 
			
			return deferred.promise;
		},
		
		/**
		 * Returns a new Promise that will automatically reject after the 
		 * specified timeout (in milliseconds) if the specified promise has 
		 * not fulfilled or rejected.
		 * 
		 * @param promiseOrValue A Promise or value.
		 * @param milliseconds Timeout duration (in milliseconds).
		 * @return Promise of the specified Promise or value that enforces the specified timeout.
		 */
		"public static function timeout",function timeout( promiseOrValue/*:**/, milliseconds/*:Number*/ )/*:Promise*/
		{
			var deferred/*:Deferred*/ = new $3.Deferred();
			
			function timerCompleteHandler()/*:void*/
			{
				timer.removeEventListener( $4.TimerEvent.TIMER_COMPLETE, timerCompleteHandler );
				
				deferred.reject( new Error( "Promise timed out." ) );
			}
			
			var timer/*:Timer*/ = new $5.Timer( Math.max( milliseconds, 0 ), 1 );
			timer.addEventListener( $4.TimerEvent.TIMER_COMPLETE, timerCompleteHandler );
			timer.start();
			
			$3.Promise.when( promiseOrValue ).then($$bound( deferred,"resolve"),$$bound( deferred,"reject") );
			
			return deferred.promise;
		},
		
		/**
		 * Traditional map function that allows input to contain Promises and/or values.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @param mapFunction Function to call to transform each resolved value in the Array.
		 * @return Promise of an Array of mapped values.
		 */
		"public static function map",function map( promisesOrValues/*:**/, mapFunction/*:Function*/ )/*:Promise*/
		{
			if ( ! (is( promisesOrValues,  Array) || $3.Promise.isThenable( promisesOrValues ) ) )
			{
				throw new Error( "Invalid parameter: expected an Array or Promise of an Array." );
			}
			if ( ! (is( mapFunction,  Function) ) )
			{
				throw new Error( "Invalid parameter: expected a function." );
			}
			
			function process( promisesOrValues/*:Array*/ )/*:Promise*/
			{
				var remainingToResolve/*:uint*/ = promisesOrValues.length;
				var results/*:Array*/ = new Array( promisesOrValues.length );
				
				var deferred/*:Deferred*/ = new $3.Deferred();
				
				if ( remainingToResolve > 0 )
				{
					function resolve( item/*:**/, index/*:uint*/ )/*:Promise*/
					{
						function transform( value/*:**/ )/*:**/
						{
							return $6.optionally( mapFunction, [ value, index, results ] );
						}
						
						function fulfill( value/*:**/ )/*:void*/
						{
							results[ index ] = value;
							if ( --remainingToResolve == 0 )
							{
								deferred.resolve( results );
							}
						}
						
						return $3.Promise.when( item ).then( transform ).then( fulfill,$$bound( deferred,"reject") );
					}
					
					for ( var index/*:uint*/ = 0; index < promisesOrValues.length; index++ )
					{
						if ( index in promisesOrValues )
						{
							resolve( promisesOrValues[ index ], index );
						}
						else
						{
							remainingToResolve--;
						}
					}
				}
				else
				{
					deferred.resolve( results );
				}
				
				return deferred.promise;
			}
			
			return $3.Promise.when( promisesOrValues ).then( process );
		},
		
		/**
		 * Traditional reduce function that allows input to contain Promises and/or values.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @param reduceFn Function to call to transform each successive item in the Array into the final reduced value.
		 * @param initialValue Initial Promise or value.
		 * @return Promise of the reduced value.
		 */
		"public static function reduce",function reduce( promisesOrValues/*:**/, reduceFunction/*:Function, ...rest*/ )/*:Promise*/
		{var rest=Array.prototype.slice.call(arguments,2);
			if ( ! (is( promisesOrValues,  Array) || $3.Promise.isThenable( promisesOrValues ) ) )
			{
				throw new Error( "Invalid parameter: expected an Array or Promise of an Array." );
			}
			if ( ! (is( reduceFunction,  Function) ) )
			{
				throw new Error( "Invalid parameter: expected a function." );
			}
			
			function reduceArray( array/*:Array*/, reduceFunction/*:Function, ...rest*/ )/*:**/
			{var rest=Array.prototype.slice.call(arguments,2);
				var index/*:uint*/ = 0;
				var length/*:uint*/ = array.length;
				var reduced/*:**/ = null;
				
				// If no initialValue, use first item of Array and adjust index to start at second item
				if ( rest.length == 0 ) {
					for ( index = 0; index < length; index++ )
					{
						if ( index in array )
						{
							reduced = array[ index ];
							index++;
							break;
						}
					}
				}
				else
				{
					reduced = rest[ 0 ];
				}
				
				while ( index < length )
				{
					if ( index in array )
					{
						reduced = reduceFunction( reduced, array[ index ], index, array );
						index++;
					}
				}
				
				return reduced;
			}
			
			function process( promisesOrValues/*:Array*/ )/*:Promise*/
			{
				// Wrap the reduce function with one that handles promises and then delegates to it.
				function reduceFnWrapper( previousValueOrPromise/*:**/, currentValueOrPromise/*:**/, currentIndex/*:uint*/, array/*:Array*/ )/*:Promise*/
				{
					function execute( previousValue/*:**/, currentValue/*:**/ )/*:**/
					{
						return $6.optionally( reduceFunction, [ previousValue, currentValue, currentIndex, array ] );
					}
					
					return $3.Promise.all( [ previousValueOrPromise, currentValueOrPromise ] ).then( $6.spread( execute ) );
				}
				
				if ( rest.length > 0 )
				{
					var initialValue/*:**/ = rest[ 0 ];
					return reduceArray( promisesOrValues, reduceFnWrapper, initialValue );
				}
				else
				{
					return reduceArray( promisesOrValues, reduceFnWrapper );
				}
			}
			
			return $3.Promise.when( promisesOrValues ).then( process );
		},
		
		/**
		 * Logs a message with the specified category, log level and optional 
		 * parameters via all registered custom logger functions.
		 * 
		 * @param category Category
		 * @param level Log level
		 * @param message Message
		 * @param parameters Optional message parameters
		 *
		 * @see #registerLogger()
		 * @see #unregisterLogger()
		 * @see com.codecatalyst.promise.logger.FlexLogger
		 * @see com.codecatalyst.promise.logger.TraceLogger
		 */
		"public static function log",function log( category/*:String*/, level/*:int*/, message/*:String, ...parameters*/ )/*:void*/
		{var parameters=Array.prototype.slice.call(arguments,3);
			var loggerParameters/*:Array*/ = [ category, level, message ].concat( parameters );
			for/* each*/ (var $2=0;$2</* in*/ $$private.loggers.length;++$2 )
			{ var logger/*:Function*/ = $$private.loggers[$2];
				logger.apply( logger, loggerParameters );
			}
		},

		/**
		 * Registers a custom adapter function capable of adapting values
		 * passed to <code>Promise.when()</code> into Promises.
		 * 
		 * A custom adapter function is called with a candidate value and
		 * should return either a Promise that adapts that value or null if the
		 * adapter cannot adapt that value.
		 * 
		 * @example A custom adapter should have the following function signature:
		 * <listing version="3.0">
		 * function function adapt( value:* ):Promise {
		 *    // ...
		 * }
		 * </listing>
		 * 
		 * @param adapter Adapter function.
		 * 
		 * @see #unregisterAdapter()
		 */
		"public static function registerAdapter",function registerAdapter( adapter/*:Function*/ )/*:void*/
		{
			if ( $$private.adapters.indexOf( adapter ) == -1 )
			{
				$$private.adapters.push( adapter );
			}
		},

		/**
		 * Unregisters a custom adapter function.
		 * 
		 * @param adapter Previously registered adapter function.
		 * 
		 * @see #registerAdapter()
		 */
		"public static function unregisterAdapter",function unregisterAdapter( adapter/*:Function*/ )/*:void*/
		{/*
			const*/var index/*:int*/ = $$private.adapters.indexOf( adapter );
			if ( index > -1 )
			{
				$$private.adapters.splice( index, 1 );
			}
		},
		
		/**
		 * Registers a custom logger function capable of logging messages
		 * with a specified category, log level, and optional parameters.
		 * 
		 * @example A custom logger should have the following function signature:
		 * <listing version="3.0">
		 * function log( category:String, level:int, message:String, ...parameters ):void {
		 *    // ...
		 * }
		 * </listing>
		 * 
		 * @param adapter Custom logger function.
		 * 
		 * @see #unregisterLogger()
		 */
		"public static function registerLogger",function registerLogger( logger/*:Function*/ )/*:void*/
		{
			if ( $$private.loggers.indexOf( logger ) == -1 )
			{
				$$private.loggers.push( logger );
			}
		},
		
		/**
		 * Unregisters a custom logger function.
		 * 
		 * @param adapter Previously registered custom logger function.
		 * 
		 * @see #registerLogger()
		 */
		"public static function unregisterLogger",function unregisterLogger( logger/*:Function*/ )/*:void*/
		{/*
			const*/var index/*:int*/ = $$private.loggers.indexOf( logger );
			if ( index > -1 )
			{
				$$private.loggers.splice( index, 1 );
			}
		},
		
		// ========================================
		// Private static methods
		// ========================================
		
		/**
		 * Schedules an Error to be rethrown in the future.
		 * 
		 * @param error Error to be thrown.
		 */
		"private static function scheduleRethrowError",function scheduleRethrowError( error/*:**/ )/*:void*/
		{
			$6.nextTick( $$private.rethrowError, [ error ] );
		},
		
		/**
		 * Rethrows the specified Error, prepending the original stack trace.
		 *  
		 * @param error Error to be thrown.
		 */
		"private static function rethrowError",function rethrowError( error/*:**/ )/*:void*/
		{
			if (is( error,  Error) )
			{
				throw error.getStackTrace() + "\nRethrown from:";
			}
			else
			{
				throw error;
			}
		},
		
		// ========================================
		// Private static properties
		// ========================================
		
		/**
		 * Array of registered adapter functions.
		 */
		"private static const",{ adapters/*:Array*/ :function(){return( []);}},
		
		/**
		 * Array of registered logger functions.
		 */
		"private static const",{ loggers/*:Array*/ :function(){return( []);}},
		
		// ========================================
		// Private properties
		// ========================================
		
		/**
		 * Internal Resolver for this Promise.
		 */
		"private var",{ resolver/*:Resolver*/:null},
		
		// ========================================
		// Constructor
		// ========================================
		
		"public function Promise",function Promise( resolver/*:Resolver*/ )
		{
			this.resolver$1 = resolver;	
		},
		
		// ========================================
		// Public methods
		// ========================================
		
		/**
		 * Attaches onFulfilled and onRejected callbacks that will be
		 * notified when the future value becomes available.
		 * 
		 * Those callbacks can subsequently transform the value that was 
		 * fulfilled or the error that was rejected. Each call to then() 
		 * returns a new Promise of that transformed value; i.e., a Promise 
		 * that is fulfilled with the callback return value or rejected with 
		 * any error thrown by the callback.
		 * 
		 * @param onFulfilled Callback to execute to transform a fulfillment value.
		 * @param onRejected Callback to execute to transform a rejection reason.
		 * 
		 * @return Promise that is fulfilled with the callback return value or rejected with any error thrown by the callback.
		 */
		"public function then",function then( onFulfilled/*:Function = null*/, onRejected/*:Function = null*/ )/*:Promise*/
		{switch(arguments.length){case 0:onFulfilled=null;case 1:onRejected=null;}
			return this.resolver$1.then( onFulfilled, onRejected );
		},
		
		/**
		 * Attaches an onRejected callback that will be notified if this
		 * Promise is rejected.
		 * 
		 * The callback can subsequently transform the reason that was 
		 * rejected. Each call to otherwise() returns a new Promise of that 
		 * transformed value; i.e., a Promise that is resolved with the 
		 * original resolved value, or resolved with the callback return value
		 * or rejected with any error thrown by the callback.
		 *
		 * @param onRejected Callback to execute to transform a rejection reason.
		 * 
		 * @return Promise of the transformed future value.
		 */
		"public function otherwise",function otherwise( onRejected/*:Function*/ )/*:Promise*/
		{
			return this.resolver$1.then( null, onRejected );
		},

		/**
		 * Attaches an onCompleted callback that will be notified when this
		 * Promise is completed.
		 * 
		 * Similar to "finally" in "try..catch..finally".
		 * 
		 * NOTE: The specified callback does not affect the resulting Promise's
		 * outcome; any return value is ignored and any Error is rethrown.
		 * 
		 * @param onCompleted Callback to execute when the Promise is resolved or rejected.
		 * 
		 * @return A new "pass-through" Promise that is resolved with the original value or rejected with the original reason.
		 */
		"public function always",function always( onCompleted/*:Function*/ )/*:Promise*/
		{
			function onFulfilled( value/*:**/ )/*:**/
			{
				try
				{
					onCompleted();
				}
				catch(error){if(is ( error,Error ))
				{
					$$private.scheduleRethrowError( error );
				}else throw error;}
				return value;
			}
			
			function onRejected( reason/*:**/ )/*:**/
			{
				try
				{
					onCompleted();
				}
				catch(error){if(is ( error,Error ))
				{
					$$private.scheduleRethrowError( error );
				}else throw error;}
				throw reason;
			}			
			
			return this.resolver$1.then( onFulfilled, onRejected );
		},
		
		/**
		 * Terminates a Promise chain, ensuring that unhandled rejections will 
		 * be rethrown as Errors.
		 * 
		 * One of the pitfalls of interacting with Promise-based APIs is the 
		 * tendency for important errors to be silently swallowed unless an 
		 * explicit rejection handler is specified.
		 * 
		 * @example For example:
		 * <listing version="3.0">
		 * promise
		 *     .then( function () {
		 *         // logic in your callback throws an error and it is interpreted as a rejection.
		 *         throw new Error("Boom!");
		 *     });
		 * // The Error was not handled by the Promise chain and is silently swallowed.
		 * </listing>
		 * 
		 * @example This problem can be addressed by terminating the Promise chain with the done() method:
		 * <listing version="3.0">
		 * promise
		 *     .then( function () {
		 *         // logic in your callback throws an error and it is interpreted as a rejection.
		 *         throw new Error("Boom!");
		 *     })
		 *     .done();
		 * // The Error was not handled by the Promise chain and is rethrown by done() on the next tick.
		 * </listing>
		 * 
		 * The done() method ensures that any unhandled rejections are rethrown 
		 * as Errors.
		 */
		"public function done",function done()/*:void*/
		{
			this.resolver$1.then( null, $$private.scheduleRethrowError );
		},
		
		/**
		 * Cancels this Promise if it is still pending, triggering a rejection 
		 * with a CancellationError that will propagate to any Promises 
		 * originating from this Promise.
		 * 
		 * NOTE: Cancellation only propagates to Promises that branch from the 
		 * target Promise. It does not traverse back up to parent branches, as 
		 * this would reject nodes from which other Promises may have branched, 
		 * causing unintended side-effects.
		 * 
		 * @param reason Cancellation reason.
		 */
		"public function cancel",function cancel( reason/*:**/ )/*:void*/
		{
			this.resolver$1.reject( new $3.CancellationError( reason ) );
		},
		
		
		/**
		 * Logs the resolution or rejection of this Promise with the specified
		 * category and optional identifier. Messages are logged via all 
		 * registered custom logger functions.
		 * 
		 * @param category Logging category, typically a class name or package.
		 * @param identifier An optional identifier to incorporate into the resulting log entry.
		 * 
		 * @return A new "pass-through" Promise that is resolved with the original value or rejected with the original reason.
		 * 
		 * @see #registerLogger()
		 * @see #unregisterLogger()
		 * @see com.codecatalyst.promise.logger.FlexLogger
		 * @see com.codecatalyst.promise.logger.TraceLogger
		 */
		"public function log",function log( category/*:String*/, identifier/*:String = null*/ )/*:Promise*/
		{if(arguments.length<=1)identifier=null;
			function onFulfilled( value/*:**/ )/*:**/
			{
				try
				{
					$3.Promise.log( category, $7.LogLevel.DEBUG, ( identifier || "Promise" ) + " resolved with value: " + value );
				}
				catch(error){if(is ( error,Error ))
				{
					$$private.scheduleRethrowError( error );
				}else throw error;}
				return value;
			}
			
			function onRejected( reason/*:**/ )/*:**/
			{
				try
				{
					$3.Promise.log( category, $7.LogLevel.ERROR, ( identifier || "Promise" ) + " rejected with reason: " + reason );
				}
				catch(error){if(is ( error,Error ))
				{
					$$private.scheduleRethrowError( error );
				}else throw error;}
				throw reason;
			}
			
			return this.resolver$1.then( onFulfilled, onRejected );
		},
	undefined];},["when","isThenable","all","any","some","delay","timeout","map","reduce","log","registerAdapter","unregisterAdapter","registerLogger","unregisterLogger"],["com.codecatalyst.promise.Deferred","Array","flash.events.TimerEvent","flash.utils.Timer","com.codecatalyst.util.optionally","com.codecatalyst.util.spread","com.codecatalyst.util.nextTick","com.codecatalyst.promise.CancellationError","com.codecatalyst.promise.logger.LogLevel"], "0.8.0", "2.0.9"
);