package com.codecatalyst.promise {

/**
 * Resolvers are used internally by Deferreds to create, resolve and reject 
	 * Promises, and to propagate fulfillment and rejection.
	 * 
	 * Developers never directly interact with a Resolver.
	 * 
	 * Each Deferred has an associated Resolver, and each Resolver has an 
	 * associated Promise. A Deferred delegates resolve() and reject() calls to 
	 * its Resolver's resolve() and reject() methods. A Promise delegates 
	 * then() calls to its Resolver's then() method. In this way, access to 
	 * Resolver operations are divided between producer (Deferred) and consumer 
	 * (Promise) roles.
	 * 
	 * When a Resolver's resolve() method is called, it fulfills with the 
	 * optionally specified value. If resolve() is called with a then-able 
	 * (i.e.a Function or Object with a then() function, such as another 
	 * Promise) it assimilates the then-able's result; the Resolver provides 
	 * its own resolve() and reject() methods as the onFulfilled or onRejected 
	 * arguments in a call to that then-able's then() function. If an error is 
	 * thrown while calling the then-able's then() function (prior to any call 
	 * back to the specified resolve() or reject() methods), the Resolver 
	 * rejects with that error. If a Resolver's resolve() method is called with 
	 * its own Promise, it rejects with a TypeError.
	 * 
	 * When a Resolver's reject() method is called, it rejects with the 
	 * optionally specified reason.
	 * 
	 * Each time a Resolver's then() method is called, it captures a pair of 
	 * optional onFulfilled and onRejected callbacks and returns a Promise of 
	 * the Resolver's future value as transformed by those callbacks.
 */
public class Resolver {
  /**
   * Promise of the future value of this Resolver.
   */
  public native function get promise():Promise;

  [ArrayElementType("com.codecatalyst.promise.Consequence")]
  public function Resolver() {
    super();
  }

  /**
   * Used to specify onFulfilled and onRejected callbacks that will be
		 * notified when the future value becomes available.
		 * 
		 * Those callbacks can subsequently transform the value that was 
		 * fulfilled or the error that was rejected. Each call to then() 
		 * returns a new Promise of that transformed value; i.e., a Promise 
		 * that is fulfilled with the callback return value or rejected with 
		 * any error thrown by the callback.
		 * 
		 * @param onFulfilled (Optional) callback to execute to transform a fulfillment value.
		 * @param onRejected (Optional) callback to execute to transform a rejection reason.
		 * 
		 * @return Promise that is fulfilled with the callback return value or rejected with any error thrown by the callback.
   */
  public native function then(onFulfilled:Function = null, onRejected:Function = null):Promise;

  /**
   * Resolve this Resolver with the (optional) specified value.
		 * 
		 * If called with a then-able (i.e.a Function or Object with a then() 
		 * function, such as another Promise) it assimilates the then-able's 
		 * result; the Resolver provides its own resolve() and reject() methods
		 * as the onFulfilled or onRejected arguments in a call to that 
		 * then-able's then() function.  If an error is  thrown while calling 
		 * the then-able's then() function (prior to any call back to the 
		 * specified resolve() or reject() methods), the Resolver rejects with 
		 * that error. If a Resolver's resolve() method is called with its own 
		 * Promise, it rejects with a TypeError.
		 * 
		 * Once a Resolver has been fulfilled or rejected, it is considered to be complete 
		 * and subsequent calls to resolve() or reject() are ignored.
		 * 
		 * @param value Value to resolve as either a fulfillment value or rejection reason.
   */
  public native function resolve(value:*):void;

  /**
   * Reject this Resolver with the specified reason.
		 * 
		 * Once a Resolver has been rejected, it is considered to be complete 
		 * and subsequent calls to resolve() or reject() are ignored.
		 * 
		 * @param reason Rejection reason.
   */
  public native function reject(reason:*):void;
}
}