package com.codecatalyst.promise {

/**
 * Promises represent a future value; i.e., a value that may not yet be available.
 */
public class Promise {
  /**
   * Returns a new Promise of the specified value, which may be an
		 * immediate value, a Promise, or a foreign Promise (i.e. Promises 
		 * from another Promises/A implementation).
		 * 
		 * Additionally, the specified value can be adapted into a Promise
		 * through the use of custom adapters.
		 * 
		 * @param value An immediate value, a Promise, a foreign Promise or adaptable value.
		 * @return Promise of the specified value.
		 * 
		 * @see #registerAdapter()
		 * @see #unregisterAdapter()
		 * @see com.codecatalyst.promise.adapter.AsyncTokenAdapter
   */
  public static native function when(value:*):Promise;

  /**
   * Determines whether the specified value is a thenable, i.e. an Object
		 * or Function that exposes a then() method and may therefore be an 
		 * third-party untrusted Promise, based on the Promises/A 
		 * specification feature test.
		 * 
		 * @param value A potential thenable.
		 * @return Boolean indicating whether the specified value was a thenable.
   */
  public static native function isThenable(value:*):Boolean;

  /**
   * Returns a new Promise that will only fulfill once all the specified
		 * Promises and/or values have been fulfilled, and will reject if any
		 * of the specified Promises is rejected. The resolution value will be 
		 * an Array containing the fulfilled value of each of the Promises or 
		 * values.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @returns Promise of an Array of the fulfilled values.
   */
  public static native function all(promisesOrValues:*):Promise;

  /**
   * Initiates a competitive race, returning a new Promise that will 
		 * fulfill when any one of the specified Promises or values is 
		 * fulfilled, or will only reject once all of the Promises have
		 * been rejected.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @return Promise of the first resolved value.
   */
  public static native function any(promisesOrValues:*):Promise;

  /**
   * Initiates a competitive race, returning a new Promise that will 
		 * fulfill when the expected number of Promises and/or values have
		 * been fulfilled, or will reject when it becomes impossible for the
		 * expected number to fulfill.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @param howMany The expected number of fulfilled values.
		 * @return Promise of the expected number of fulfilled values.
   */
  public static native function some(promisesOrValues:*, howMany:uint):Promise;

  /**
   * Returns a new Promise that will automatically resolve with the 
		 * specified Promise or value after the specified delay 
		 * (in milliseconds).
		 *
		 * @param promiseOrValue A Promise or value.
		 * @param milliseconds Delay duration (in milliseconds).
		 * @return Promise of the specified Promise or value that will resolve after the specified delay.
   */
  public static native function delay(promiseOrValue:*, milliseconds:Number):Promise;

  /**
   * Returns a new Promise that will automatically reject after the 
		 * specified timeout (in milliseconds) if the specified promise has 
		 * not fulfilled or rejected.
		 * 
		 * @param promiseOrValue A Promise or value.
		 * @param milliseconds Timeout duration (in milliseconds).
		 * @return Promise of the specified Promise or value that enforces the specified timeout.
   */
  public static native function timeout(promiseOrValue:*, milliseconds:Number):Promise;

  /**
   * Traditional map function that allows input to contain Promises and/or values.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @param mapFunction Function to call to transform each resolved value in the Array.
		 * @return Promise of an Array of mapped values.
   */
  public static native function map(promisesOrValues:*, mapFunction:Function):Promise;

  /**
   * Traditional reduce function that allows input to contain Promises and/or values.
		 * 
		 * @param promisesOrValues An Array of values or Promises, or a Promise of an Array of values or Promises.
		 * @param reduceFn Function to call to transform each successive item in the Array into the final reduced value.
		 * @param initialValue Initial Promise or value.
		 * @return Promise of the reduced value.
   */
  public static native function reduce(promisesOrValues:*, reduceFunction:Function, ...rest):Promise;

  /**
   * Logs a message with the specified category, log level and optional 
		 * parameters via all registered custom logger functions.
		 * 
		 * @param category Category
		 * @param level Log level
		 * @param message Message
		 * @param parameters Optional message parameters
		 *
		 * @see #registerLogger()
		 * @see #unregisterLogger()
		 * @see com.codecatalyst.promise.logger.FlexLogger
		 * @see com.codecatalyst.promise.logger.TraceLogger
   */
  public static native function log(category:String, level:int, message:String, ...parameters):void;

  /**
   * Registers a custom adapter function capable of adapting values
		 * passed to <code>Promise.when()</code> into Promises.
		 * 
		 * A custom adapter function is called with a candidate value and
		 * should return either a Promise that adapts that value or null if the
		 * adapter cannot adapt that value.
		 * 
		 * @example A custom adapter should have the following function signature:
		 * <listing version="3.0">
		 * function function adapt( value:* ):Promise {
		 *    // ...
		 * }
		 * </listing>
		 * 
		 * @param adapter Adapter function.
		 * 
		 * @see #unregisterAdapter()
   */
  public static native function registerAdapter(adapter:Function):void;

  /**
   * Unregisters a custom adapter function.
		 * 
		 * @param adapter Previously registered adapter function.
		 * 
		 * @see #registerAdapter()
   */
  public static native function unregisterAdapter(adapter:Function):void;

  /**
   * Registers a custom logger function capable of logging messages
		 * with a specified category, log level, and optional parameters.
		 * 
		 * @example A custom logger should have the following function signature:
		 * <listing version="3.0">
		 * function log( category:String, level:int, message:String, ...parameters ):void {
		 *    // ...
		 * }
		 * </listing>
		 * 
		 * @param adapter Custom logger function.
		 * 
		 * @see #unregisterLogger()
   */
  public static native function registerLogger(logger:Function):void;

  /**
   * Unregisters a custom logger function.
		 * 
		 * @param adapter Previously registered custom logger function.
		 * 
		 * @see #registerLogger()
   */
  public static native function unregisterLogger(logger:Function):void;

  public function Promise(resolver:Resolver) {
    super();
  }

  /**
   * Attaches onFulfilled and onRejected callbacks that will be
		 * notified when the future value becomes available.
		 * 
		 * Those callbacks can subsequently transform the value that was 
		 * fulfilled or the error that was rejected. Each call to then() 
		 * returns a new Promise of that transformed value; i.e., a Promise 
		 * that is fulfilled with the callback return value or rejected with 
		 * any error thrown by the callback.
		 * 
		 * @param onFulfilled Callback to execute to transform a fulfillment value.
		 * @param onRejected Callback to execute to transform a rejection reason.
		 * 
		 * @return Promise that is fulfilled with the callback return value or rejected with any error thrown by the callback.
   */
  public native function then(onFulfilled:Function = null, onRejected:Function = null):Promise;

  /**
   * Attaches an onRejected callback that will be notified if this
		 * Promise is rejected.
		 * 
		 * The callback can subsequently transform the reason that was 
		 * rejected. Each call to otherwise() returns a new Promise of that 
		 * transformed value; i.e., a Promise that is resolved with the 
		 * original resolved value, or resolved with the callback return value
		 * or rejected with any error thrown by the callback.
		 *
		 * @param onRejected Callback to execute to transform a rejection reason.
		 * 
		 * @return Promise of the transformed future value.
   */
  public native function otherwise(onRejected:Function):Promise;

  /**
   * Attaches an onCompleted callback that will be notified when this
		 * Promise is completed.
		 * 
		 * Similar to "finally" in "try..catch..finally".
		 * 
		 * NOTE: The specified callback does not affect the resulting Promise's
		 * outcome; any return value is ignored and any Error is rethrown.
		 * 
		 * @param onCompleted Callback to execute when the Promise is resolved or rejected.
		 * 
		 * @return A new "pass-through" Promise that is resolved with the original value or rejected with the original reason.
   */
  public native function always(onCompleted:Function):Promise;

  /**
   * Terminates a Promise chain, ensuring that unhandled rejections will 
		 * be rethrown as Errors.
		 * 
		 * One of the pitfalls of interacting with Promise-based APIs is the 
		 * tendency for important errors to be silently swallowed unless an 
		 * explicit rejection handler is specified.
		 * 
		 * @example For example:
		 * <listing version="3.0">
		 * promise
		 *     .then( function () {
		 *         // logic in your callback throws an error and it is interpreted as a rejection.
		 *         throw new Error("Boom!");
		 *     });
		 * // The Error was not handled by the Promise chain and is silently swallowed.
		 * </listing>
		 * 
		 * @example This problem can be addressed by terminating the Promise chain with the done() method:
		 * <listing version="3.0">
		 * promise
		 *     .then( function () {
		 *         // logic in your callback throws an error and it is interpreted as a rejection.
		 *         throw new Error("Boom!");
		 *     })
		 *     .done();
		 * // The Error was not handled by the Promise chain and is rethrown by done() on the next tick.
		 * </listing>
		 * 
		 * The done() method ensures that any unhandled rejections are rethrown 
		 * as Errors.
   */
  public native function done():void;

  /**
   * Cancels this Promise if it is still pending, triggering a rejection 
		 * with a CancellationError that will propagate to any Promises 
		 * originating from this Promise.
		 * 
		 * NOTE: Cancellation only propagates to Promises that branch from the 
		 * target Promise. It does not traverse back up to parent branches, as 
		 * this would reject nodes from which other Promises may have branched, 
		 * causing unintended side-effects.
		 * 
		 * @param reason Cancellation reason.
   */
  public native function cancel(reason:*):void;

  /**
   * Logs the resolution or rejection of this Promise with the specified
		 * category and optional identifier. Messages are logged via all 
		 * registered custom logger functions.
		 * 
		 * @param category Logging category, typically a class name or package.
		 * @param identifier An optional identifier to incorporate into the resulting log entry.
		 * 
		 * @return A new "pass-through" Promise that is resolved with the original value or rejected with the original reason.
		 * 
		 * @see #registerLogger()
		 * @see #unregisterLogger()
		 * @see com.codecatalyst.promise.logger.FlexLogger
		 * @see com.codecatalyst.promise.logger.TraceLogger
   */
  public native function log(category:String, identifier:String = null):Promise;
}
}