/*
 * Copyright 1999-2011 Alibaba Group.
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *      http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.dubbo.remoting.transport.grizzly;

import java.io.IOException;

import org.glassfish.grizzly.Connection;
import org.glassfish.grizzly.filterchain.BaseFilter;
import org.glassfish.grizzly.filterchain.FilterChainContext;
import org.glassfish.grizzly.filterchain.NextAction;

import com.alibaba.dubbo.common.URL;
import com.alibaba.dubbo.remoting.RemotingException;

import lombok.extern.slf4j.Slf4j;
import net.jahhan.common.extension.utils.StringUtils;
import net.jahhan.spi.ChannelHandler;

/**
 * GrizzlyHandler
 * 
 * @author william.liangf
 */
@Slf4j
public class GrizzlyHandler extends BaseFilter {

    private final URL url;
    
    private final ChannelHandler handler;
    
    public GrizzlyHandler(URL url, ChannelHandler handler){
        this.url = url;
        this.handler = handler;
    }

    @Override
    public NextAction handleConnect(FilterChainContext ctx) throws IOException {
        Connection<?> connection = ctx.getConnection();
        GrizzlyChannel channel = GrizzlyChannel.getOrAddChannel(connection, url, handler);
        try {
            handler.connected(channel);
        } catch (RemotingException e) {
            throw new IOException(StringUtils.toString(e));
        } finally {
            GrizzlyChannel.removeChannelIfDisconnectd(connection);
        }
        return ctx.getInvokeAction();
    }
    
    @Override
    public NextAction handleClose(FilterChainContext ctx) throws IOException {
        Connection<?> connection = ctx.getConnection();
        GrizzlyChannel channel = GrizzlyChannel.getOrAddChannel(connection, url, handler);
        try {
            handler.disconnected(channel);
        } catch (RemotingException e) {
            throw new IOException(StringUtils.toString(e));
        } finally {
            GrizzlyChannel.removeChannelIfDisconnectd(connection);
        }
        return ctx.getInvokeAction();
    }

    @Override
    public NextAction handleRead(FilterChainContext ctx) throws IOException {
        Connection<?> connection = ctx.getConnection();
        GrizzlyChannel channel = GrizzlyChannel.getOrAddChannel(connection, url, handler);
        try {
            handler.received(channel, ctx.getMessage());
        } catch (RemotingException e) {
            throw new IOException(StringUtils.toString(e));
        } finally {
            GrizzlyChannel.removeChannelIfDisconnectd(connection);
        }
        return ctx.getInvokeAction();
    }

    @Override
    public NextAction handleWrite(FilterChainContext ctx) throws IOException {
        Connection<?> connection = ctx.getConnection();
        GrizzlyChannel channel = GrizzlyChannel.getOrAddChannel(connection, url, handler);
        try {
            handler.sent(channel, ctx.getMessage());
        } catch (RemotingException e) {
            throw new IOException(StringUtils.toString(e));
        } finally {
            GrizzlyChannel.removeChannelIfDisconnectd(connection);
        }
        return ctx.getInvokeAction();
    }
    
    @Override
    public void exceptionOccurred(FilterChainContext ctx, Throwable error) {
        Connection<?> connection = ctx.getConnection();
        GrizzlyChannel channel = GrizzlyChannel.getOrAddChannel(connection, url, handler);
        try {
            handler.caught(channel, error);
        } catch (RemotingException e) {
            log.error("RemotingException on channel " + channel, e);
        } finally {
            GrizzlyChannel.removeChannelIfDisconnectd(connection);
        }
    }
    
}