package com.thebluekernel.kmmcommons.websocket

import platform.Foundation.*
import platform.darwin.NSObject

/**
 * Created by Ahmed Ibrahim on 24/07/2022
 */
internal actual class WebSocketClient actual constructor(){
    private var events: WebSocketListener? = null
    private var webSocket: NSURLSessionWebSocketTask? = null
    private var retryCount: Int = 3

    actual fun connect(request: WSRequest, listener: WebSocketListener) {
        this.events = listener
        events?.onConnecting()
        val socketEndpoint = NSURL.URLWithString(request.url)!!
        val socketRequest = NSMutableURLRequest.requestWithURL(socketEndpoint)
        request.headers.forEach { socketRequest.setValue(it.value, it.key) }
        val urlSession = NSURLSession.sessionWithConfiguration(
            configuration = NSURLSessionConfiguration.defaultSessionConfiguration(),
            delegate = object : NSObject(), NSURLSessionWebSocketDelegateProtocol {
                override fun URLSession(
                    session: NSURLSession,
                    webSocketTask: NSURLSessionWebSocketTask,
                    didOpenWithProtocol: String?
                ) {
                    retryCount = 3
                    events?.onOpen()
                }

                override fun URLSession(
                    session: NSURLSession,
                    task: NSURLSessionTask,
                    didCompleteWithError: NSError?
                ) {
                    val errorMessage =
                        didCompleteWithError?.localizedDescription ?: "Closed with error."
                    events?.onError(Throwable(errorMessage))
                }
            },
            delegateQueue = NSOperationQueue.currentQueue()
        )

        webSocket = urlSession.webSocketTaskWithRequest(socketRequest)
        listenMessages()
        webSocket?.resume()
    }

    actual fun send(message: String) {
        webSocket?.sendMessage(NSURLSessionWebSocketMessage(message)) { err ->
            val errorMessage = err?.localizedDescription ?: "Error sending message."
            events?.onError(Throwable(errorMessage))
        }
    }

    actual fun close() {
        webSocket?.cancelWithCloseCode(NSURLSessionWebSocketCloseCodeNormalClosure, null)
        webSocket = null
        events?.onClose(1_000, "Normal closing.")
    }

    private fun listenMessages() {
        webSocket?.receiveMessageWithCompletionHandler { message, nsError ->
            when {
                nsError != null -> {
                    val errorMessage = nsError.localizedDescription ?: "Error receiving message."
                    events?.onError(Throwable(errorMessage))
                }
                message != null -> {
                    message.string?.let { events?.onMessage(it) }
                    retryCount = 3
                }
            }
            if (retryCount >= 0) {
                listenMessages()
                retryCount--
            }
        }
    }

}