package com.thebluekernel.kmmcommons.websocket

import kotlinx.coroutines.flow.MutableSharedFlow

/**
 * Created by Ahmed Ibrahim on 23/07/2022
 */
internal abstract class WebSocketListener {
    open fun onConnecting() {}
    open fun onOpen() {}
    open fun onMessage(message: String) {}
    open fun onClosing(code: Int, reason: String) {}
    open fun onClose(code: Int, reason: String) {}
    open fun onError(error: Throwable) {}
}

private const val EVENTS_BUFFER_SIZE = 100

internal class DefaultWebSocketEventObserver : WebSocketListener() {
    private val _eventsFlow =
        MutableSharedFlow<WebSocketEvent>(extraBufferCapacity = EVENTS_BUFFER_SIZE)

    val eventsFlow = _eventsFlow

    override fun onConnecting() {
        _eventsFlow.tryEmit(WebSocketEvent.Connecting)
    }

    override fun onOpen() {
        _eventsFlow.tryEmit(WebSocketEvent.Connected)
    }

    override fun onMessage(message: String) {
        _eventsFlow.tryEmit(WebSocketEvent.Message(message))
    }

    override fun onClosing(code: Int, reason: String) {
        _eventsFlow.tryEmit(WebSocketEvent.Closing(code, reason))
    }

    override fun onClose(code: Int, reason: String) {
        _eventsFlow.tryEmit(WebSocketEvent.Closed(code, reason))
    }

    override fun onError(error: Throwable) {
        _eventsFlow.tryEmit(WebSocketEvent.Error(error))
    }

}

