package com.thebluekernel.kmmcommons.network

import com.thebluekernel.kmmcommons.logging.error
import com.thebluekernel.kmmcommons.logging.initLogger
import io.github.aakira.napier.Napier
import io.ktor.client.call.*
import io.ktor.client.plugins.*
import io.ktor.client.plugins.contentnegotiation.*
import io.ktor.client.plugins.logging.*
import io.ktor.client.request.*
import io.ktor.client.request.forms.*
import io.ktor.client.statement.*
import io.ktor.http.*
import io.ktor.serialization.kotlinx.json.*
import kotlinx.serialization.json.Json

/**
 * Created by Ahmed Ibrahim on 09,May,2022
 */

/**
 * Helper function to provide HttpClient with Kotlin DSL
 */
inline fun buildHttpClient(config: HttpConfig.() -> Unit): AppHttpClient {
    val httpConfig = HttpConfig()
    httpConfig.config()
    return AppHttpClient(httpConfig)
}

class AppHttpClient(private val config: HttpConfig) {
    val baseUrl: String

    val client by lazy {
        httpClient {
            // Enable logs if passed
            val logLevel = when (config.httpLogLevel) {
                HttpLogLevel.NONE -> LogLevel.NONE
                HttpLogLevel.HEADERS -> LogLevel.HEADERS
                HttpLogLevel.BODY -> LogLevel.BODY
                HttpLogLevel.ALL -> LogLevel.ALL
            }
            install(Logging) {
                level = logLevel
                logger = object : Logger {
                    override fun log(message: String) {
                        val tag = "HTTP Client"
                        when (config.loggerLevel) {
                            LoggerLevel.DEBUG -> Napier.d(tag = tag, message = message)
                            LoggerLevel.VERBOSE -> Napier.v(tag = tag, message = message)
                            LoggerLevel.ERROR -> Napier.e(tag = tag, message = message)
                        }
                    }
                }
            }

            // Serialization
            install(ContentNegotiation) {
                json(
                    Json {
                        isLenient = true
                        prettyPrint = true
                        ignoreUnknownKeys = true
                    }
                )
            }

            // Timeout
            install(HttpTimeout) {
                requestTimeoutMillis = config.requestTimeoutMillis
                connectTimeoutMillis = config.connectTimeoutMillis
                socketTimeoutMillis = config.socketTimeoutMillis
            }

            // Caching
            if (config.enableCaching) {
                install(CachePlugin) {
                    findCacheEntry = config.cachePluginConfig.findCacheEntry
                    storeCacheEntry = config.cachePluginConfig.storeCacheEntry
                    deleteCacheEntry = config.cachePluginConfig.deleteCacheEntry
                    maxAge = config.cachePluginConfig.maxAge
                }
            }

            // Defaults
            defaultRequest {
                config.headers.forEach { entry ->
                    header(entry.key, entry.value)
                }
            }

        }.also { initLogger() }
    }

    init {
        requireNotNull(config.baseUrl) { "BaseUrl must not be null!" }
        this.baseUrl = config.baseUrl.orEmpty()
    }

    suspend inline fun <reified T : Any> get(
        path: String,
        vararg params: CallParam,
        headers: Map<String, String> = mapOf()
    ): ResponseResult<T> = runCatching {
        val response = client.get(baseUrl.plus(path)) {
            params.forEach { param -> parameter(param.key, param.value) }
            headers.forEach { header -> header(header.key, header.value) }
        }
        ResponseResult.success(process<T>(response))
    }.getOrElse {
        error("Error Calling network", it)
        ResponseResult.error(it)
    }

    suspend inline fun <reified T : Any> post(
        path: String,
        body: Any,
        headers: Map<String, String> = mapOf()
    ): ResponseResult<T> = runCatching {
        val response = client.post(baseUrl.plus(path)) {
            contentType(ContentType.Application.Json)
            headers.forEach { header -> header(header.key, header.value) }
            setBody(body)
        }
        ResponseResult.success(process<T>(response))
    }.getOrElse { ResponseResult.error(it) }

    suspend inline fun <reified T : Any> patch(
        path: String,
        body: Any,
        headers: Map<String, String> = mapOf()
    ): ResponseResult<T> = runCatching {
        val response = client.patch(baseUrl.plus(path)) {
            contentType(ContentType.Application.Json)
            headers.forEach { header -> header(header.key, header.value) }
            setBody(body)
        }
        ResponseResult.success(process<T>(response))
    }.getOrElse { ResponseResult.error(it) }

    suspend inline fun <reified T : Any> put(
        path: String,
        body: Any,
        headers: Map<String, String> = mapOf()
    ): ResponseResult<T> = runCatching {
        val response = client.put(baseUrl.plus(path)) {
            contentType(ContentType.Application.Json)
            headers.forEach { header -> header(header.key, header.value) }
            setBody(body)
        }
        ResponseResult.success(process<T>(response))
    }.getOrElse {
        ResponseResult.error(it)
    }

    suspend inline fun <reified T : Any> delete(
        path: String,
        vararg params: CallParam,
        headers: Map<String, String> = mapOf()
    ): ResponseResult<T> = runCatching {
        val response = client.delete(baseUrl.plus(path)) {
            params.forEach { param -> parameter(param.key, param.value) }
            headers.forEach { header -> header(header.key, header.value) }
        }
        ResponseResult.success(process<T>(response))
    }.getOrElse {
        error("Error Calling network", it)
        ResponseResult.error(it)
    }

    suspend inline fun <reified T : Any> upload(
        path: String,
        content: MultipartContent,
        headers: Map<String, String> = mapOf(),
        crossinline progressCallback: (total: Long, sent: Long) -> Unit
    ): ResponseResult<T> = runCatching {
        val response = client.post(baseUrl.plus(path)) {
            headers.forEach { header -> header(header.key, header.value) }
            setBody(MultiPartFormDataContent(formData {
                content.params.forEach { append(it.key, it.value) }
                append(content.name, content.content, Headers.build {
                    append(HttpHeaders.ContentType, content.mimeType)
                })
            }))
            onUpload { bytesSentTotal, contentLength ->
                progressCallback.invoke(contentLength, bytesSentTotal)
            }
        }
        ResponseResult.success(process<T>(response))
    }.getOrElse { ResponseResult.error(it) }

    suspend inline fun <reified T : Any> process(response: HttpResponse): T {
        return when (response.status.isSuccess()) {
            true -> response.body() as T
            false -> throw NetworkException(response.status.value, response.bodyAsText())
        }
    }
}

sealed class ResponseResult<T : Any> {
    data class Success<T : Any>(val data: T) : ResponseResult<T>()
    data class Error<T : Any>(val error: Throwable) : ResponseResult<T>()

    companion object {
        fun <T : Any> success(data: T) = Success(data)
        fun <T : Any> error(error: Throwable) = Error<T>(error)
    }
}

data class CallParam(val key: String, val value: Any)

data class MultipartContent(
    val name: String,
    val content: ByteArray,
    val mimeType: String,
    val params: Map<String, String> = mapOf()
) {
    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || this::class != other::class) return false

        other as MultipartContent

        if (name != other.name) return false
        if (!content.contentEquals(other.content)) return false

        return true
    }

    override fun hashCode(): Int {
        var result = name.hashCode()
        result = 31 * result + content.contentHashCode()
        return result
    }
}