package net.jackadull.specdriven.adapter.scalatest

import java.util.concurrent.atomic.AtomicBoolean

import net.jackadull.specdriven.Specification
import net.jackadull.specdriven.requirement.Requirement
import org.scalatest.{BeforeAndAfterAllConfigMap, ConfigMap, FreeSpec}

import scala.language.implicitConversions
import scala.util.Try

trait SpecificationForScalaTest extends FreeSpec with BeforeAndAfterAllConfigMap {
  def specifications:Seq[Specification[()=>Unit]]

  implicit def forScalaTest[O](specification:Specification[O])(implicit assert:ScalaTestAssertOutcome[O]):Specification[()=>Unit] =
    specification.map {o => {() => assert(o)}}

  private lazy val cachedSpecifications:Seq[Specification[()=>Unit]] = specifications
  private var reqsToCleanUp:Set[Requirement[()=>Unit]] = Set()

  cachedSpecifications foreach {spec =>
    s"[Specification] ${spec.specificationTitle}" - {
      spec.requirements.foreach {req =>
        s"[Requirement] ${req.requirementTitle}: ${req.givenAndWhenSetup}" - {
          lazy val performResult:Try[Unit] = Try {cleanUpLater(req); req.perform()}
          req.expectations.foreach {expectation =>
            expectation.description.in {
              performResult
              ScalaTestAssertOutcome(expectation, acceptFeatureNotImplemented.get, performResult)
            }
          }
        }
      }
    }
  }

  protected def cleanUpLater(req:Requirement[()=>Unit]):Unit = synchronized {reqsToCleanUp = reqsToCleanUp + req}

  private val acceptFeatureNotImplemented = new AtomicBoolean(true)

  override protected def beforeAll(configMap:ConfigMap):Unit =
    acceptFeatureNotImplemented.set {configMap.get("acceptFeatureNotImplemented") match {
      case Some(nope) if Set("false", "no", "not", "0").contains(s"$nope".toLowerCase) => false
      case _ => true
    }}

  override protected def afterAll(configMap:ConfigMap):Unit = {
    val errorEncountered = new AtomicBoolean()
    var goOn = true
    do {
      val nextReq:Option[Requirement[()=>Unit]] = synchronized {
        if(reqsToCleanUp.isEmpty) None else {
          val req = reqsToCleanUp.head
          reqsToCleanUp = reqsToCleanUp - req
          Some(req)
        }
      }
      nextReq match {
        case None => goOn = false
        case Some(req) =>
          try {req.cleanUp()}
          catch {case x:Throwable =>
            Console.err.println(s"----- Exception during cleanUp() of requirement: ${req.requirementTitle} (${req.completePhrase})")
            x.printStackTrace(Console.err)
            Console.err.println("-----")
            errorEncountered.set(true)
          }
      }
    } while(goOn)
    if(errorEncountered.get) fail("at least one error encountered during cleanUp(); see error output for details")
  }
}
