/*
 * Copyright (c) 2003, 2004, 2005, 2006 Israfil Consulting Services Corporation
 * Copyright (c) 2003, 2004, 2005, 2006 Christian Edward Gruber
 * All Rights Reserved
 * 
 * This software is licensed under the Berkeley Standard Distribution license,
 * (BSD license), as defined below:
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this 
 *    list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Israfil Consulting Services nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 * 
 * $Id: ArrayUtils.java 48 2006-02-23 06:11:53Z cgruber $
 */
package net.israfil.foundation.collections;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import net.israfil.foundation.core.Copyright;

/**
 * An array utility class.  This class is generic, so its usage is similar
 * to the following:
 * Foo[] foo1 = new Foo[]{ aFoo, bFoo, cFoo };
 * Foo[] foo2 = new Foo[]{ dFoo, eFoo, fFoo };
 * Foo[] fooResult = ArrayUtils.add(new Foo[],foo1,foo2);
 * 
 * fooResult would be equivalent: new Foo[]{ aFoo, bFoo, cFoo, dFoo, eFoo, fFoo }
 * 
 * This util is mostly helpful for putting together large batches of like
 * arrays into a long single array, such as large groups of parameter lists
 * or whatever. 
 * 
 * @author <a href="mailto:cgruber@israfil.net">Christian Edward Gruber </a>
 */
@Copyright(years={"2003","2004","2005","2006"},owner="Israfil Consulting Services Corporation",license="BSD")
public class ArrayUtils {
	private static final Logger logger = Logger.getLogger(ArrayUtils.class.getName());

    /**
     * Adds arrays of like type creating a new array containing
     * all elements of the added arrays.  Note that you have to
     * pass in a typed array, because you can't create an array
     * from a generic type in the code.
     * 
     * i.e. you can't do this with Generics.
     * <pre>
     *   public class Foo<T> {
     *       public T[] getArray() {
     *   		T[] array = new T[a];
     * 	     }
     *   }
     * </pre>
     * 
     * However, as described in the tests, you can simply pass
     * it an empty array of that type.
     * 
     * @param resultBucket An array of the type you are attempting to aggregate.  Can be empty.  If it has sufficient space, it will be used for the aggregated result.
     * @param arrays The arrays to be aggregated.
     * @return An array of the provided type, containing the aggregated contents of the provided arrays.
     */
    public static <E> E[] aggregate(E[] resultBucket, E[] ... arrays) 
    {
    	logger.log(Level.FINER,Array.newInstance(arrays.getClass().getComponentType(),0).getClass().getCanonicalName());
    	List<E> resultList = new ArrayList<E>();
    	for (E[] t : arrays) {
    		for (E element : t) {
    			resultList.add(element);
    		}
    	}
    	logger.log(Level.FINER,resultList.toString());    	
    	return (E[])resultList.toArray(resultBucket);
    }


}

