package net.ifok.common.net;

import net.ifok.common.io.IOUtils;
import net.ifok.common.net.model.HttpResponse;
import net.ifok.common.string.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @Description:  http client operation
 * @Author: leftso
 * @Date: 2021/1/13 10:15
 **/
public class HttpClientUtils {
    /**
     * 单例
     */
    private static CloseableHttpClient singletonHttpClient=null;
    private static PoolingHttpClientConnectionManager cm;
    /**
     * client
     */
    private static HttpClient httpClient=null;
    /**
     * 单位毫秒 60s
     */
    private static final int MAX_SOCKET_TIMEOUT = 60000;
    /**
     * 单位毫秒 60s
     */
    private static final int MAX_CONNECTION_TIMEOUT = 60000;
    /**
     * config public
     */
    private static final RequestConfig DEFAULT_CONFIG = RequestConfig.custom().setSocketTimeout(MAX_SOCKET_TIMEOUT).setConnectTimeout(MAX_CONNECTION_TIMEOUT).build();


    static {
        cm=new PoolingHttpClientConnectionManager();
        //每个路由的最大并发数连接
        cm.setDefaultMaxPerRoute(10);
        //总连接数
        cm.setMaxTotal(30);
        singletonHttpClient=HttpClients.custom().setConnectionManager(cm)
                .setMaxConnTotal(400).setMaxConnPerRoute(150)
                .setDefaultRequestConfig(DEFAULT_CONFIG)
                //新开线程清理过期的连接 解决NoHttpResponseException 异常
                .evictExpiredConnections()
                .build();
    }

    private HttpClientUtils(){}

    /**
     * 获取client
     * @return
     */
    public synchronized static HttpClient  getHttpClient() throws IOException {
        return singletonHttpClient;
    }


    /**
     * post请求 json参数
     *
     * @param url
     * @param bodyJsonParams
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doPost(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpPost httpPost = new HttpPost(url);
        httpPost.addHeader("Content-Type", "application/json");
        httpPost.setEntity(new StringEntity(bodyJsonParams, Charset.forName("UTF-8")));

        addHeader(httpPost, headers);
        return execute(httpPost, config);
    }

    /**
     * POST请求,json参数
     * @param url
     * @param bodyJsonParams
     * @param headers
     * @param timeoutSec
     * @param proxyHost
     * @param proxyPort
     * @param proxyScheme
     * @return
     * @throws IOException
     */
    public static HttpResponse doPost(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme) throws IOException {
        RequestConfig config = RequestConfig.custom()
                .setSocketTimeout(timeoutSec * 1000)
                .setConnectTimeout(timeoutSec * 1000)
                .setProxy(getProxy(proxyHost,proxyPort,proxyScheme))
                .build();
        HttpPost httpPost = new HttpPost(url);
        httpPost.addHeader("Content-Type", "application/json");
        httpPost.setEntity(new StringEntity(bodyJsonParams, Charset.forName("UTF-8")));

        addHeader(httpPost, headers);
        return execute(httpPost, config);
    }

    /**
     * post请求 json参数
     *
     * @param url
     * @param bodyJsonParams
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doPost(String url, String bodyJsonParams, Map<String, String> headers) throws IOException {
        HttpPost httpPost = new HttpPost(url);
        httpPost.addHeader("Content-Type", "application/json");
        httpPost.setEntity(new StringEntity(bodyJsonParams, Charset.forName("UTF-8")));
        addHeader(httpPost, headers);
        return execute(httpPost);
    }

    /**
     * post k-v参数
     *
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doPost(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpPost httpPost = new HttpPost(url);
        if (params != null && params.keySet().isEmpty()) {
            httpPost.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPost, headers);
        return execute(httpPost, config);
    }

    /**
     * POST K-V参数
     * @param url
     * @param params
     * @param headers
     * @param timeoutSec
     * @param proxyHost
     * @param proxyPort
     * @param proxyScheme
     * @return
     * @throws IOException
     */
    public static HttpResponse doPost(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme) throws IOException {
        RequestConfig config = RequestConfig.custom()
                .setSocketTimeout(timeoutSec * 1000)
                .setConnectTimeout(timeoutSec * 1000)
                .setProxy(getProxy(proxyHost,proxyPort,proxyScheme))
                .build();
        HttpPost httpPost = new HttpPost(url);
        if (params != null && params.keySet().isEmpty()) {
            httpPost.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPost, headers);
        return execute(httpPost, config);
    }


    /**
     * post k-v参数
     *
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doPost(String url, Map<String, String> params, Map<String, String> headers)
            throws IOException {
        HttpPost httpPost = new HttpPost(url);
        if (params != null && params.keySet().isEmpty()) {
            httpPost.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPost, headers);
        return execute(httpPost);
    }

    /**
     * patch json参数
     *
     * @param url
     * @param bodyJsonParams
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doPatch(String url, String bodyJsonParams, Map<String, String> headers) throws IOException {
        HttpPatch httpPatch = new HttpPatch(url);
        httpPatch.setEntity(new StringEntity(bodyJsonParams));
        addHeader(httpPatch, headers);
        return execute(httpPatch);
    }

    public static HttpResponse doPatch(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpPatch httpPatch = new HttpPatch(url);
        httpPatch.setEntity(new StringEntity(bodyJsonParams));
        addHeader(httpPatch, headers);
        return execute(httpPatch, config);
    }

    public static HttpResponse doPatch(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec , String proxyHost, Integer proxyPort, String proxyScheme) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpPatch httpPatch = new HttpPatch(url);
        httpPatch.setEntity(new StringEntity(bodyJsonParams));
        addHeader(httpPatch, headers);
        return execute(httpPatch, config);
    }


    /**
     * patch k-v参数
     *
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doPatch(String url, Map<String, String> params, Map<String, String> headers)
            throws IOException {
        HttpPatch httpPatch = new HttpPatch(url);
        if (params != null && !params.isEmpty()) {
            httpPatch.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPatch, headers);
        return execute(httpPatch);
    }

    public static HttpResponse doPatch(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpPatch httpPatch = new HttpPatch(url);
        if (params != null && !params.isEmpty()) {
            httpPatch.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPatch, headers);
        return execute(httpPatch, config);
    }
    public static HttpResponse doPatch(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec , String proxyHost, Integer proxyPort, String proxyScheme)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpPatch httpPatch = new HttpPatch(url);
        if (params != null && !params.isEmpty()) {
            httpPatch.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPatch, headers);
        return execute(httpPatch, config);
    }
    /**
     * PUT JSON参数
     *
     * @param url
     * @param bodyJsonParams
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doPut(String url, String bodyJsonParams, Map<String, String> headers) throws IOException {
        HttpPut httpPut = new HttpPut(url);
        httpPut.addHeader("Content-Type", "application/json");
        httpPut.setEntity(new StringEntity(bodyJsonParams, Charset.forName("UTF-8")));

        addHeader(httpPut, headers);
        return execute(httpPut);
    }

    public static HttpResponse doPut(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpPut httpPut = new HttpPut(url);
        httpPut.addHeader("Content-Type", "application/json");
        httpPut.setEntity(new StringEntity(bodyJsonParams, Charset.forName("UTF-8")));

        addHeader(httpPut, headers);
        return execute(httpPut, config);
    }
    public static HttpResponse doPut(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec , String proxyHost, Integer proxyPort, String proxyScheme) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpPut httpPut = new HttpPut(url);
        httpPut.addHeader("Content-Type", "application/json");
        httpPut.setEntity(new StringEntity(bodyJsonParams, Charset.forName("UTF-8")));

        addHeader(httpPut, headers);
        return execute(httpPut, config);
    }

    /**
     * put k-v参数
     *
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doPut(String url, Map<String, String> params, Map<String, String> headers)
            throws IOException {
        HttpPut httpPut = new HttpPut(url);
        if (params != null && params.keySet().isEmpty()) {
            httpPut.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPut, headers);
        return execute(httpPut);
    }

    public static HttpResponse doPut(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpPut httpPut = new HttpPut(url);
        if (params != null && params.keySet().isEmpty()) {
            httpPut.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPut, headers);
        return execute(httpPut, config);
    }
    public static HttpResponse doPut(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpPut httpPut = new HttpPut(url);
        if (params != null && params.keySet().isEmpty()) {
            httpPut.setEntity(getUrlEncodedFormEntity(params));
        }
        addHeader(httpPut, headers);
        return execute(httpPut, config);
    }
    /**
     * Delete json 参数
     *
     * @param url
     * @param bodyJsonParams
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doDeletedoPut(String url, String bodyJsonParams, Map<String, String> headers) throws IOException {

        HttpDeleteWithEntity httpDelete = new HttpDeleteWithEntity(url);
        httpDelete.setEntity(new StringEntity(bodyJsonParams));
        addHeader(httpDelete, headers);
        return execute(httpDelete);
    }

    public static HttpResponse doDeletedoPut(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpDeleteWithEntity httpDelete = new HttpDeleteWithEntity(url);
        httpDelete.setEntity(new StringEntity(bodyJsonParams));
        addHeader(httpDelete, headers);
        return execute(httpDelete, config);
    }
    public static HttpResponse doDeletedoPut(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpDeleteWithEntity httpDelete = new HttpDeleteWithEntity(url);
        httpDelete.setEntity(new StringEntity(bodyJsonParams));
        addHeader(httpDelete, headers);
        return execute(httpDelete, config);
    }
    /**
     * delete k-v参数
     *
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doDelete(String url, Map<String, String> params, Map<String, String> headers)
            throws IOException {
        HttpDeleteWithEntity httpDelete = new HttpDeleteWithEntity(url);
        addHeader(httpDelete, headers);
        if (params != null && !params.isEmpty()) {
            httpDelete.setEntity(getUrlEncodedFormEntity(params));
        }
        return execute(httpDelete);
    }

    public static HttpResponse doDelete(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpDeleteWithEntity httpDelete = new HttpDeleteWithEntity(url);
        addHeader(httpDelete, headers);
        if (params != null && !params.isEmpty()) {
            httpDelete.setEntity(getUrlEncodedFormEntity(params));
        }
        return execute(httpDelete, config);
    }

    public static HttpResponse doDelete(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpDeleteWithEntity httpDelete = new HttpDeleteWithEntity(url);
        addHeader(httpDelete, headers);
        if (params != null && !params.isEmpty()) {
            httpDelete.setEntity(getUrlEncodedFormEntity(params));
        }
        return execute(httpDelete, config);
    }
    /**
     * options json参数
     *
     * @param url
     * @param bodyJsonParams
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doOptions(String url, String bodyJsonParams, Map<String, String> headers) throws IOException {
        HttpOptionsWithEntity httpOptions = new HttpOptionsWithEntity(url);
        addHeader(httpOptions, headers);
        httpOptions.setEntity(new StringEntity(bodyJsonParams));
        return execute(httpOptions);
    }

    public static HttpResponse doOptions(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpOptionsWithEntity httpOptions = new HttpOptionsWithEntity(url);
        addHeader(httpOptions, headers);
        httpOptions.setEntity(new StringEntity(bodyJsonParams));
        return execute(httpOptions, config);
    }
    public static HttpResponse doOptions(String url, String bodyJsonParams, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpOptionsWithEntity httpOptions = new HttpOptionsWithEntity(url);
        addHeader(httpOptions, headers);
        httpOptions.setEntity(new StringEntity(bodyJsonParams));
        return execute(httpOptions, config);
    }

    /**
     * options k-v参数
     *
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doOptions(String url, Map<String, String> params, Map<String, String> headers)
            throws IOException {
        HttpOptionsWithEntity httpOptions = new HttpOptionsWithEntity(url);
        addHeader(httpOptions, headers);
        if (params != null && !params.isEmpty()) {
            httpOptions.setEntity(getUrlEncodedFormEntity(params));
        }
        return execute(httpOptions);
    }

    public static HttpResponse doOptions(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpOptionsWithEntity httpOptions = new HttpOptionsWithEntity(url);
        addHeader(httpOptions, headers);
        if (params != null && !params.isEmpty()) {
            httpOptions.setEntity(getUrlEncodedFormEntity(params));
        }
        return execute(httpOptions, config);
    }
    public static HttpResponse doOptions(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme)
            throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpOptionsWithEntity httpOptions = new HttpOptionsWithEntity(url);
        addHeader(httpOptions, headers);
        if (params != null && !params.isEmpty()) {
            httpOptions.setEntity(getUrlEncodedFormEntity(params));
        }
        return execute(httpOptions, config);
    }

    /**
     * head请求
     *
     * @param url
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doHeader(String url, Map<String, String> headers) throws IOException {
        HttpHead httpHead = new HttpHead(url);
        addHeader(httpHead, headers);
        return execute(httpHead);

    }

    public static HttpResponse doHeader(String url, Map<String, String> headers, int timeoutSec) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        HttpHead httpHead = new HttpHead(url);
        addHeader(httpHead, headers);
        return execute(httpHead, config);
    }
    public static HttpResponse doHeader(String url, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        HttpHead httpHead = new HttpHead(url);
        addHeader(httpHead, headers);
        return execute(httpHead, config);
    }


    /**
     * get请求
     *
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doGet(String url, Map<String, String> params, Map<String, String> headers) throws IOException {
        return doGet(url,params,headers,MAX_CONNECTION_TIMEOUT);
    }

    public static HttpResponse doGet(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec) throws IOException {
        return doGet(url,params,headers,timeoutSec,null);
    }

    public static HttpResponse doGet(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec,String responseCharset) throws IOException{
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
        // 参数
        StringBuilder paramsBuilder = new StringBuilder(url);

        if (params != null && !params.keySet().isEmpty()) {
            if (!url.contains("?")) {
                paramsBuilder.append("?");
            }
            List<NameValuePair> list = new ArrayList<>();

            Set<String> keySet = params.keySet();
            Iterator<String> iterator = keySet.iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                String value = params.get(key);
                list.add(new BasicNameValuePair(key, value));
            }
            String paramsStr = EntityUtils.toString(new UrlEncodedFormEntity(list));
            paramsBuilder.append(paramsStr);
        }
        HttpGet httpGet = new HttpGet(paramsBuilder.toString());
        // 头
        addHeader(httpGet, headers);

        return execute(httpGet,config,responseCharset);
    }

    public static HttpResponse doGet(String url, Map<String, String> params, Map<String, String> headers, int timeoutSec, String proxyHost, Integer proxyPort, String proxyScheme) throws IOException {
        RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).setProxy(getProxy(proxyHost,proxyPort,proxyScheme)).build();
        // params

        StringBuilder paramsBuilder = new StringBuilder(url);
        if (params != null && !params.keySet().isEmpty()) {
            if (!url.contains("?")) {
                paramsBuilder.append("?");
            }
            List<NameValuePair> list = new ArrayList<>();

            Set<String> keySet = headers.keySet();
            Iterator<String> iterator = keySet.iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                String value = headers.get(key);
                list.add(new BasicNameValuePair(key, value));
            }
            String paramsStr = EntityUtils.toString(new UrlEncodedFormEntity(list));
            paramsBuilder.append(paramsStr);
        }
        HttpGet httpGet = new HttpGet(paramsBuilder.toString());
        // 头
        addHeader(httpGet, headers);
        return execute(httpGet,config);
    }

    /**
     * 下载图片,返回的事base64编码后的字符串
     *  请使用doGetFile替换
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    @Deprecated
    public static HttpResponse doGetImg(String url, Map<String, String> params, Map<String, String> headers) throws IOException {
        // 参数
        StringBuilder paramsBuilder = new StringBuilder(url);

        if (params != null && params.keySet().isEmpty()) {
            if (url.indexOf("?") == -1) {
                paramsBuilder.append("?");
            }
            List<NameValuePair> list = new ArrayList<>();

            Set<String> keySet = params.keySet();
            Iterator<String> iterator = keySet.iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                String value = params.get(key);
                list.add(new BasicNameValuePair(key, value));
            }
            String paramsStr = EntityUtils.toString(new UrlEncodedFormEntity(list));
            paramsBuilder.append(paramsStr);
        }
        HttpGet httpGet = new HttpGet(paramsBuilder.toString());
        // 头
        addHeader(httpGet, headers);
        try (CloseableHttpClient httpClient = HttpClients.createDefault()) {
            CloseableHttpResponse response = httpClient.execute(httpGet);
            System.out.println(response.getStatusLine().getStatusCode());
            InputStream is = response.getEntity().getContent();
            byte[] imgBytes = IOUtils.inputStream2byte(is);
            return new HttpResponse(Base64.getEncoder().encodeToString(imgBytes), response.getStatusLine().getStatusCode());
        }
    }

    /**
     * 文件下载
     * @param url
     * @param params
     * @param headers
     * @return
     * @throws IOException
     */
    public static HttpResponse doGetFile(String url, Map<String, String> params, Map<String, String> headers) throws IOException {
        HttpResponse httpResponse = doGetFile(url, params, headers);
        return httpResponse;
    }
    /**
     * 下载文件
     * @param url 请求地址
     * @param params 参数列表
     * @param headers 请求头部
     * @param timeoutSec 超时秒数
     * @return
     * @throws IOException
     */
    public static HttpResponse doGetFile(String url, Map<String, String> params, Map<String, String> headers,Integer timeoutSec) throws IOException {
        // 参数
        StringBuilder paramsBuilder = new StringBuilder(url);

        if (params != null && params.keySet().isEmpty()) {
            if (url.indexOf("?") == -1) {
                paramsBuilder.append("?");
            }
            List<NameValuePair> list = new ArrayList<>();

            Set<String> keySet = params.keySet();
            Iterator<String> iterator = keySet.iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                String value = params.get(key);
                list.add(new BasicNameValuePair(key, value));
            }
            String paramsStr = EntityUtils.toString(new UrlEncodedFormEntity(list));
            paramsBuilder.append(paramsStr);
        }
        HttpGet httpGet = new HttpGet(paramsBuilder.toString());
        // 头
        addHeader(httpGet, headers);

        try (CloseableHttpClient httpClient = HttpClients.createDefault()) {
            CloseableHttpResponse response;
            if (Objects.nonNull(timeoutSec)){
                RequestConfig config = RequestConfig.custom().setSocketTimeout(timeoutSec * 1000).setConnectTimeout(timeoutSec * 1000).build();
                httpGet.setConfig(config);
                response = httpClient.execute(httpGet);
            }else{
                httpGet.setConfig(DEFAULT_CONFIG);
                response = httpClient.execute(httpGet);
            }
            InputStream is = response.getEntity().getContent();
            byte[] imgBytes = IOUtils.inputStream2byte(is);
            return new HttpResponse(Base64.getEncoder().encodeToString(imgBytes), response.getStatusLine().getStatusCode());
        }
    }


    private static HttpResponse execute(HttpRequestBase httpRequestBase) throws IOException {
        return execute(httpRequestBase, null);
    }

    /**
     * 执行请求并返回string值
     *
     * @param httpRequestBase
     * @return
     * @throws IOException
     */
    private static HttpResponse execute(HttpRequestBase httpRequestBase, RequestConfig config) throws IOException {
        return  execute(httpRequestBase,config,null);
    }
    private static HttpResponse execute(HttpRequestBase httpRequestBase, RequestConfig config,String responseCharset) throws IOException {
        if (config == null) {
            config = DEFAULT_CONFIG;
        }
        httpRequestBase.setConfig(config);

        HttpClient httpClient = getHttpClient();
        org.apache.http.HttpResponse response = httpClient.execute(httpRequestBase);
        System.out.println(response.getStatusLine().getStatusCode());
        Header type = response.getEntity().getContentType();
        String defaultCharset = "UTF-8";
        String charset = getCharSet(type.getValue());
        if (charset != null && !"".equals(charset)) {
            defaultCharset = charset;
        }
        if (!StringUtils.isEmpty(responseCharset)){
            defaultCharset=responseCharset;
        }
        return new HttpResponse(EntityUtils.toString(response.getEntity(), defaultCharset), response.getStatusLine().getStatusCode());
    }

    /**
     * 添加请求头部
     *
     * @param httpUriRequest
     * @param headers
     */
    private static void addHeader(HttpUriRequest httpUriRequest, Map<String, String> headers) {
        if (httpUriRequest != null) {
            if (headers != null && !headers.keySet().isEmpty()) {
                Set<String> keySet = headers.keySet();
                Iterator<String> iterator = keySet.iterator();
                while (iterator.hasNext()) {
                    String key = iterator.next();
                    String value = headers.get(key);
                    httpUriRequest.addHeader(key, value);
                }
            }
        }
    }

    /**
     * 获取 UrlEncodedFormEntity 参数实体
     *
     * @param params
     * @return
     * @throws UnsupportedEncodingException
     */
    private static UrlEncodedFormEntity getUrlEncodedFormEntity(Map<String, String> params) throws UnsupportedEncodingException {
        if (params != null && params.keySet().isEmpty()) {
            List<NameValuePair> list = new ArrayList<>();

            Set<String> keySet = params.keySet();
            Iterator<String> iterator = keySet.iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                String value = params.get(key);
                list.add(new BasicNameValuePair(key, value));
            }
            return new UrlEncodedFormEntity(list);
        }
        return null;
    }

    private static HttpHost getProxy(String proxyHost, Integer proxyPort, String proxyScheme){
        HttpHost proxy=null;
        if (!StringUtils.isEmpty(proxyHost)) {
            if (proxyPort != null && !StringUtils.isEmpty(proxyScheme)) {
                proxy = new HttpHost(proxyHost, proxyPort, proxyScheme);
            } else if (proxyPort != null && StringUtils.isEmpty(proxyScheme)) {
                proxy = new HttpHost(proxyHost, proxyPort);
            } else {
                proxy = new HttpHost(proxyHost);
            }
        }
        return proxy;
    }
    /**
     * 根据HTTP 响应头部的content type抓取响应的字符集编码
     *
     * @param content
     * @return
     */
    private static String getCharSet(String content) {
        String regex = ".*charset=([^;]*).*";
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(content);
        if (matcher.find()){
            return matcher.group(1);
        }else{
            return null;
        }

    }


    /**
     * 解决httpclient 的DELETE默认不支持setEntity
     */
    static class HttpDeleteWithEntity extends HttpEntityEnclosingRequestBase {
        public static final String METHOD_NAME = "DELETE";

        @Override
        public String getMethod() {
            return METHOD_NAME;
        }

        public HttpDeleteWithEntity(final String uri) {
            super();
            setURI(URI.create(uri));
        }

        public HttpDeleteWithEntity(final URI uri) {
            super();
            setURI(uri);
        }

        public HttpDeleteWithEntity() {
            super();
        }
    }

    /**
     * 解决httpclient 的OPTIONS默认不支持setEntity
     */
    static class HttpOptionsWithEntity extends HttpEntityEnclosingRequestBase {
        public static final String METHOD_NAME = "OPTIONS";

        @Override
        public String getMethod() {
            return METHOD_NAME;
        }

        public HttpOptionsWithEntity() {
            super();
        }

        public HttpOptionsWithEntity(final String uri) {
            super();
            setURI(URI.create(uri));
        }

        public HttpOptionsWithEntity(final URI uri) {
            super();
            setURI(uri);
        }
    }

    /**
     * 获取重定向URL地址
     * @param url 重定向后的URL地址
     * @return
     * @throws IOException
     */
    public static String getRedirectURL(String url,Map<String,String> headers) throws IOException {
        RequestConfig requestConfig=RequestConfig.custom().setRedirectsEnabled(false).build();
        try (CloseableHttpClient httpClient= HttpClients.custom().setDefaultRequestConfig(requestConfig).build();){
            HttpGet httpGet = new HttpGet(url);
            addHeader(httpGet,headers);

            CloseableHttpResponse response = httpClient.execute(httpGet);
            if (response.getStatusLine().getStatusCode()==302){
                Header location = response.getFirstHeader("location");
                return location.getValue();
            }
            return url;
        }
    }


}
