package net.ifok.common.image;

import net.coobird.thumbnailator.Thumbnails;
import net.coobird.thumbnailator.geometry.Positions;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

/**
 * @Description:  图片压缩
 * @Author: xq 
 * @Date: 2021/1/13 17:06
 **/
public class ImageCompressUtils {

    /**
     * 流方式输入输出图片压缩
     *
     * @param src     文件输入流（单个）
     * @param target  压缩后文件输出流（单个）
     * @param quality 压缩质量
     * @throws IOException
     */
    public static void compress(InputStream src, OutputStream target, float quality) throws IOException {
        Thumbnails.of(src).scale(1)
                .outputQuality(quality).toOutputStream(target);
    }

    /**
     * 流方式输入输出图片压缩
     *
     * @param src     文件输入流（单个）
     * @param target  压缩后文件输出流（单个）
     * @param quality 压缩质量
     * @throws IOException
     */
    public static void compress(InputStream src, OutputStream target, float quality, String format) throws IOException {
        Thumbnails.of(src).scale(1)
                //输出格式
                .outputFormat(format)
                .outputQuality(quality).toOutputStream(target);
    }

    /**
     * 压缩并重置大小(流式)
     *
     * @param src
     * @param target
     * @param quality
     * @param width
     * @param height
     * @throws IOException
     */
    public static void compressAndReSize(InputStream src, OutputStream target, float quality, int width, int height) throws IOException {
        Thumbnails.of(src).size(width, height).keepAspectRatio(false).outputQuality(quality).toOutputStream(target);
    }

    /**
     * @param src
     * @param target
     * @param quality
     * @param width
     * @param height
     * @param format
     * @throws IOException
     */
    public static void compressAndReSize(InputStream src, OutputStream target, float quality, int width, int height, String format) throws IOException {
        Thumbnails.of(src).size(width, height).keepAspectRatio(false)
                //输出格式
                .outputFormat(format)
                .outputQuality(quality).toOutputStream(target);
    }

    /**
     * 根据宽比例压缩(流式)
     *
     * @param src
     * @param target
     * @param quality
     * @param width
     * @throws IOException
     */
    public static void compressAndSizeWidth(InputStream src, OutputStream target, float quality, int width) throws IOException {
        Thumbnails.of(src).width(width).keepAspectRatio(true).outputQuality(quality).toOutputStream(target);
    }

    /**
     * 根据宽比例压缩(流式)
     *
     * @param src
     * @param target
     * @param quality
     * @param width
     * @throws IOException
     */
    public static void compressAndSizeWidth(InputStream src, OutputStream target, float quality, int width, String format) throws IOException {
        Thumbnails.of(src).width(width).keepAspectRatio(true)
                //输出格式
                .outputFormat(format)
                .outputQuality(quality).toOutputStream(target);
    }


    /**
     * 根据宽比例压缩(流式)
     *
     * @param src
     * @param target
     * @param quality
     * @param height
     * @throws IOException
     */
    public static void compressAndSizeHeight(InputStream src, OutputStream target, float quality, int height) throws IOException {
        Thumbnails.of(src).height(height).keepAspectRatio(true).outputQuality(quality).toOutputStream(target);
    }

    /**
     * 根据宽比例压缩(流式)
     *
     * @param src
     * @param target
     * @param quality
     * @param height
     * @throws IOException
     */
    public static void compressAndSizeHeight(InputStream src, OutputStream target, float quality, int height, String format) throws IOException {
        Thumbnails.of(src).height(height).keepAspectRatio(true)
                //输出格式
                .outputFormat(format)
                .outputQuality(quality).toOutputStream(target);
    }

    /**
     * 压缩并添加图片水印(流式)
     *
     * @param src      原图
     * @param target   输出图
     * @param mark     水印图片
     * @param position 水印位置
     * @param opacity  水印透明度 0.0f-1.0f
     * @param quality  图片质量 0-1
     * @throws IOException
     */
    public static void compressAndWatermark(InputStream src, OutputStream target, InputStream mark, Positions position, float opacity, float quality) throws IOException {
        Thumbnails.of(src)
                .scale(1)
                .watermark(position, ImageIO.read(mark), opacity)
                .outputQuality(quality)
                .toOutputStream(target);
    }

    /**
     * 压缩并添加图片水印(流式)
     *
     * @param src      原图
     * @param target   输出图
     * @param mark     水印图片
     * @param position 水印位置
     * @param opacity  水印透明度 0.0f-1.0f
     * @param quality  图片质量 0-1
     * @throws IOException
     */
    public static void compressAndWatermark(InputStream src, OutputStream target, InputStream mark, Positions position, float opacity, float quality, String format) throws IOException {
        Thumbnails.of(src).scale(1)
                .watermark(position, ImageIO.read(mark), opacity)
                //输出格式
                .outputFormat(format)
                .outputQuality(quality).toOutputStream(target);
    }

    public static void compressAndWatermarks(InputStream src, OutputStream target, InputStream mark, List<Positions> positions, float opacity, float quality, String format) throws IOException {
        Thumbnails.Builder builder= Thumbnails.of(src).scale(1);
        BufferedImage ico=ImageIO.read(mark);
        for (Positions p:positions) {
            builder=builder.watermark(p,ico , opacity);
        }
        //输出格式
        builder.outputFormat(format)
                .outputQuality(quality).toOutputStream(target);
    }
}
