package net.ifok.common.security;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;

/**
 * @Description:  DES加解密工具类
 * @Author: xq 
 * @Date: 2021/1/17 21:37
 **/
public class DesUtils {
    /** 默认算法格式 **/
    final static String default_transformation = "DES/ECB/PKCS5Padding";

    /**
     * 根据key获取加密器
     *
     * @param keyData
     *            key 8byte
     * @return 加密器
     * @throws Exception
     *             异常
     */
    private static Cipher getEncryptCipher(byte[] keyData) throws Exception {
        if (keyData.length != 8) {
            throw new Exception("Key Data Must 8 byte!");
        }
        SecretKeySpec key = new SecretKeySpec(keyData, "DES");
        // 指定分块ECB模式，填充PKCS5Padding模式
        Cipher encryptCipher = Cipher.getInstance(default_transformation);
        // 初始化加密的容器
        encryptCipher.init(Cipher.ENCRYPT_MODE, key);
        return encryptCipher;
    }

    /**
     * 根据key获取解码器
     *
     * @return 解码器
     * @throws Exception
     *             异常
     */
    private static Cipher getDecryptCipher(byte[] keyData) throws Exception {
        if (keyData.length != 8) {
            throw new Exception("Key Data Must 8 byte!");
        }
        SecretKeySpec key = new SecretKeySpec(keyData, "DES");
        Cipher decryptCipher = Cipher.getInstance(default_transformation);
        // 初始化解密的容器
        decryptCipher.init(Cipher.DECRYPT_MODE, key);
        return decryptCipher;
    }

    /**
     * DES加密
     *
     * @param data
     *            待加密数据
     * @param keyData
     *            key值
     * @return 加密后的数据
     * @throws Exception
     *             异常
     */
    public static byte[] encrypt(byte[] data, byte[] keyData) throws Exception {
        return getEncryptCipher(keyData).doFinal(data);
    }

    /**
     * DES解密
     *
     * @param data
     *            加密后的数据
     *
     * @param keyData
     *            key值
     * @return 解密数据
     * @throws Exception
     *             异常
     */
    public static byte[] decrypt(byte[] data, byte[] keyData) throws Exception {
        return getDecryptCipher(keyData).doFinal(data);
    }

    /**
     * 测试
     *
     * @param args
     */
    public static void main(String[] args) {
        try {
            byte[] data = "测试123456".getBytes();
            byte[] keyData = "12345678".getBytes();
            System.out.println("原文:" + new String(data));
            byte[] enData = encrypt(data, keyData);
            System.out.println("加密后:" + new String(enData));
            byte[] deData = decrypt(enData, keyData);
            System.out.println("解密后:" + new String(deData));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
