package net.ifok.common.base;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

/**
 * 集合/对象属性抽取，用于节约宽带或加强数据安全性
 */
public class BeanAttrFilterUtils {

    /**
     * 从普通POJO类中提取关键信息, 要求遵循 getMethod 规范
     *
     * @param target       目标JAVA POJO类,要求装有数据
     * @param targetFields 类的属性名字: 希望从 POJO 中提取哪些字段的数据
     * @return 数据会被装载于MAP中
     */
    public static Map<String, Object> extractAttrFromPojo(Object target, String... targetFields) {
        if ((null == target) || null == targetFields || 0 == targetFields.length) {
            return new HashMap<>();
        }
        Map<String, Object> objectMap = new HashMap<>();
        for (String targetField : targetFields) {
            Class<?> targetClass = target.getClass();
            try {
                Field field = targetClass.getDeclaredField(targetField);
                Object invoke = invokeGetMethod(target, targetClass, field, targetField);
                objectMap.put(targetField, invoke);
            } catch (Exception e) {
                System.out.println(e);
            }
        }
        return objectMap;
    }

    /**
     * 方法说明: 过滤掉不要的属性
     */
    public static Map<String, Object> removeAttrFromPojo(Object target, String... targetFields) {
        if ((null == target) || null == targetFields || 0 == targetFields.length) {
            return new HashMap<>();
        }
        Map<String, Object> objectMap = new HashMap<>();
        Class<?> targetClass = target.getClass();
        Field[] fields = targetClass.getDeclaredFields();
        List<String> listStrings = Arrays.asList(targetFields);
        for (Field field : fields) {
            String targetField = field.getName();
            String s = listStrings.stream().filter(v->v.equals(targetField)).findFirst().orElse(null);
            if(s == null){
                try {
                    Object invoke= invokeGetMethod(target, targetClass, field, targetField);
                    objectMap.put(targetField,invoke);
                } catch (Exception e) {
                    System.out.println(e);
                }
            }
        }
        return objectMap;
    }


    public static void removeValueForCollection(Collection<?> targets, String... targetFields){
        if (Objects.nonNull(targets)&&targetFields.length>0){
            for (Object target : targets) {
                removeValueFormPojo(target,targetFields);
            }
        }
    }

    public static void removeValueFormPojo(Object target,String... targetFields){
        if (Objects.nonNull(target)&&targetFields.length>0){
            Class<?> targetClass = target.getClass();
            Field[] fields = targetClass.getDeclaredFields();
            List<String> fieldNames = Arrays.asList(targetFields);
            for (Field field : fields) {
                String s = fieldNames.stream().filter(o -> Objects.equals(o, field.getName())).findFirst().orElse(null);
                if (Objects.nonNull(s)){
                    try {
                        invokeSetMethod(target,targetClass,field,s,null);
                    }catch (Exception e) {
                        System.out.println(e);
                    }
                }
            }

        }
    }

    private static Object invokeGetMethod(Object target, Class<?> targetClass, Field field, String targetField) throws IllegalAccessException, InvocationTargetException, IntrospectionException {
        PropertyDescriptor pd = new PropertyDescriptor(field.getName(), targetClass);
        Method readMethod = pd.getReadMethod();
        Object invoke = readMethod.invoke(target);
        return invoke;
    }


    /**
     * 方法说明: 调用设置方法
     * @author xqlee
     */
    private static void invokeSetMethod(Object target, Class<?> targetClass, Field field, String targetField,Object targetValue) throws IllegalAccessException, InvocationTargetException, IntrospectionException {
        if (!field.getType().getName().equals("boolean")){
            PropertyDescriptor pd=new PropertyDescriptor(field.getName(),targetClass);
            Method writeMethod = pd.getWriteMethod();
            writeMethod.invoke(target,new Object[]{targetValue});
        }
    }


    /**
     * 从集合中过滤掉不需要字段
     *
     * 适用于：list<对象>
     *
     * @param targets
     * @param targetFields
     * @return
     */
    public static List<Map<String, Object>> removeAttrForCollection(Collection<?> targets, String... targetFields) {
        List<Map<String, Object>> result = new ArrayList<>();
        if (Objects.nonNull(targets)&&targetFields.length>0){
            for (Object target : targets) {
                result.add(removeAttrFromPojo(target, targetFields));
            }
        }
        return result;
    }

    /**
     * 从集合中提取信息
     *
     * 适用于：list<对象>
     *
     * @param targets
     * @param targetFields
     * @return
     */
    public static List<Map<String, Object>> extractAttrForCollection(Collection<?> targets, String... targetFields) {
        List<Map<String, Object>> result = new ArrayList<>();
        if (Objects.nonNull(targets)&&targetFields.length>0){
            for (Object target : targets) {
                result.add(extractAttrFromPojo(target, targetFields));
            }
        }

        return result;
    }
}
