package net.wysistat.sdk.operation

import kotlinx.coroutines.*
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.flow.receiveAsFlow

internal class OperationQueue(private val scope: CoroutineScope = CoroutineScope(Dispatchers.Default)) {
    private val pendingChannel = Channel<Operation>()
    private var isCancelling = false

    init {
        scope.launch {
            pendingChannel
                .receiveAsFlow()
                .collect { operation ->
                    if(!isCancelling) {
                        operation.execute()
                    } else if(operation is CancellingOperation) {
                        // When we get the cancel operation, we indicate to our flow that we can execute next operations
                        isCancelling = false
                    }
                }
        }
    }

    /**
     * Method to add operation to the OperationQueue.
     * Firstly, we switch to our scope dedicated to manage operation.
     * Then we add our operation with a suspend function
     *
     * @param operation the operation to add
     */
    fun enqueue(operation: Operation) = runBlocking {
        scope.launch {
            enqueueOperation(operation)
        }
    }

    private suspend fun enqueueOperation(operation: Operation) {
        pendingChannel.send(operation)
    }

    /**
     * Cancel all pending operations. A custom [CancellingOperation] will be added in the Channel.
     * All operation until this Operation will be cancelled
     */
    fun cancelPending() {
        isCancelling = true
        enqueue(CancellingOperation())
    }
}
