package net.wysistat.sdk

import android.annotation.SuppressLint
import android.content.Context
import androidx.annotation.Keep
import net.wysistat.sdk.operation.Operation
import net.wysistat.sdk.operation.OperationQueue

/**
 * Main interface of the Wysistat SDK
 * This interface provides all necessary methods to initialize the SDK and send tracking events
 */
@Keep
interface Wysistat {
    /**
     * Initializes the SDK with a context and an account
     * This method must be called before any other use of the SDK
     *
     * @param context The Android context
     * @param account The Wysistat account identifier
     * @return A Result containing the initialization result
     *
     * On success, returns Result.success
     * On error, returns Result.failure with the corresponding error (see [WysistatError])
     */
    @Keep
    fun init(context: Context, account: String): Result<Unit>

    /**
     * Checks if the SDK is initialized
     * This method allows verifying if the SDK has been properly initialized before using it
     *
     * @return true if the SDK is initialized, false otherwise
     */
    @Keep
    fun isInit(): Boolean

    /**
     * Sets the list of sub-accounts to use
     * Sub-accounts allow segmenting tracking data
     * This list will be used for all subsequent events until modified or cleared
     *
     * @param subAccounts List of sub-account identifiers
     */
    @Keep
    fun setSubAccounts(subAccounts: List<String>)

    /**
     * Clears the list of sub-accounts
     * After calling this method, events will be sent without sub-accounts
     */
    @Keep
    fun clearSubAccounts()

    /**
     * Sends a page view event
     * This method uses the default configuration (no sub-accounts, no extranet counter)
     *
     * @param page The name of the visited page
     * @return A Result containing the sending result
     *
     * On success, returns Result.success(true)
     * On error, returns Result.failure with the corresponding error (see [WysistatError])
     */
    @Keep
    fun send(
        page: String
    ): Result<Boolean>

    /**
     * Sends a page view event with specific sub-accounts
     * This method allows specifying sub-accounts only for this event,
     * without modifying the global configuration
     *
     * @param page The name of the visited page
     * @param subAccounts List of sub-accounts for this event
     * @return A Result containing the sending result
     *
     * On success, returns Result.success(true)
     * On error, returns Result.failure with the corresponding error (see [WysistatError])
     */
    @Keep
    fun send(
        page: String,
        subAccounts: List<String>
    ): Result<Boolean>

    /**
     * Sends a page view event with an extranet counter
     * The extranet counter allows tracking specific statistics for certain users
     *
     * @param page The name of the visited page
     * @param counterExtranet The extranet counter identifier
     * @return A Result containing the sending result
     *
     * On success, returns Result.success(true)
     * On error, returns Result.failure with the corresponding error (see [WysistatError])
     */
    @Keep
    fun send(
        page: String,
        counterExtranet: String
    ): Result<Boolean>

    /**
     * Sends a page view event with an extranet counter and specific sub-accounts
     * This method combines the use of an extranet counter and specific sub-accounts
     *
     * @param page The name of the visited page
     * @param counterExtranet The extranet counter identifier
     * @param subAccounts List of sub-accounts for this event
     * @return A Result containing the sending result
     *
     * On success, returns Result.success(true)
     * On error, returns Result.failure with the corresponding error (see [WysistatError])
     */
    @Keep
    fun send(
        page: String,
        counterExtranet: String,
        subAccounts: List<String>
    ): Result<Boolean>

    @Keep
    companion object : Wysistat {

        /**
         * Operation queue to execute
         * This queue manages events asynchronously and in order
         */
        private val executionQueue = OperationQueue()

        /**
         * Limit of stored events
         * This limit prevents excessive memory consumption
         */
        internal const val limiteStock = 500

        /**
         * Unique instance of the SDK
         * This instance is initialized when calling init()
         */
        @SuppressLint("StaticFieldLeak")
        @Volatile
        @Keep
        private lateinit var instance: WysistatImpl

        /**
         * Initializes the SDK with a context and an account
         * This method can only be called once
         *
         * @param context The Android context
         * @param account The Wysistat account identifier
         * @return A Result containing the initialization result
         *
         * On success, returns Result.success
         * On error, returns Result.failure with the corresponding error (see [WysistatError])
         */
        @Keep
        override fun init(context: Context, account: String): Result<Unit> {
            if (this::instance.isInitialized) {
                return Result.failure(WysistatError.AlreadyActivated)
            }

            instance = WysistatImpl(context, account)

            return Result.success(Unit)
        }

        /**
         * Checks if the SDK is initialized
         * This method should be used before calling methods requiring initialization
         *
         * @return true if the SDK is initialized, false otherwise
         */
        @Keep
        override fun isInit(): Boolean {
            return this::instance.isInitialized
        }

        /**
         * Sets the list of global sub-accounts to use
         *
         * @param subAccounts List of sub-account identifiers
         * @throws WysistatError.NotActivated if the SDK is not initialized
         */
        @Keep
        override fun setSubAccounts(subAccounts: List<String>) {
            instance.setSubAccounts(subAccounts)
        }

        /**
         * Clears the list of global sub-accounts
         *
         * @throws WysistatError.NotActivated if the SDK is not initialized
         */
        @Keep
        override fun clearSubAccounts() {
            instance.clearSubAccounts()
        }

        /**
         * Sends a simple page view event
         *
         * @param page The name of the visited page
         * @return A Result containing the sending result
         *
         * On success, returns Result.success(true)
         * On error, returns Result.failure with the corresponding error (see [WysistatError])
         */
        @Keep
        override fun send(
            page: String
        ): Result<Boolean> {

            if (!this::instance.isInitialized) {
                return Result.failure(WysistatError.NotActivated)
            }

            executionQueue.enqueue(
                operation = Operation(actionBlock = {
                    instance.send(
                        page = page
                    )
                })
            )

            return Result.success(true)
        }

        /**
         * Sends a page view event with specific sub-accounts
         *
         * @param page The name of the visited page
         * @param subAccounts List of sub-accounts for this event
         * @return A Result containing the sending result
         *
         * On success, returns Result.success(true)
         * On error, returns Result.failure with the corresponding error (see [WysistatError])
         */
        @Keep
        override fun send(
            page: String,
            subAccounts: List<String>
        ): Result<Boolean> {
            if (!this::instance.isInitialized) {
                return Result.failure(WysistatError.NotActivated)
            }

            executionQueue.enqueue(
                operation = Operation(actionBlock = {
                    instance.send(
                        page = page,
                        subAccounts = subAccounts
                    )
                })
            )

            return Result.success(true)
        }

        /**
         * Sends a page view event with an extranet counter
         *
         * @param page The name of the visited page
         * @param counterExtranet The extranet counter identifier
         * @return A Result containing the sending result
         *
         * On success, returns Result.success(true)
         * On error, returns Result.failure with the corresponding error (see [WysistatError])
         */
        @Keep
        override fun send(
            page: String,
            counterExtranet: String
        ): Result<Boolean> {

            if (!this::instance.isInitialized) {
                return Result.failure(WysistatError.NotActivated)
            }

            executionQueue.enqueue(
                operation = Operation(actionBlock = {
                    instance.send(
                        page = page,
                        counterExtranet = counterExtranet
                    )
                })
            )

            return Result.success(true)
        }

        /**
         * Sends a page view event with an extranet counter and specific sub-accounts
         *
         * @param page The name of the visited page
         * @param counterExtranet The extranet counter identifier
         * @param subAccounts List of sub-accounts for this event
         * @return A Result containing the sending result
         *
         * On success, returns Result.success(true)
         * On error, returns Result.failure with the corresponding error (see [WysistatError])
         */
        @Keep
        override fun send(
            page: String,
            counterExtranet: String,
            subAccounts: List<String>
        ): Result<Boolean> {

            if (!this::instance.isInitialized) {
                return Result.failure(WysistatError.NotActivated)
            }

            executionQueue.enqueue(
                operation = Operation(actionBlock = {
                    instance.send(
                        page = page,
                        counterExtranet = counterExtranet,
                        subAccounts = subAccounts
                    )
                })
            )

            return Result.success(true)
        }
    }
}