package googleapis.firebase

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.client.Client

class ProjectsAndroidAppsClient[F[_]: Concurrent](client: Client[F])
    extends AbstractClient[F](client) {
  val baseUri = uri"https://firebase.googleapis.com/"
  def undelete(
      /** Required. The resource name of the AndroidApp, in the format: projects/ PROJECT_IDENTIFIER/androidApps/APP_ID Since an APP_ID is a unique identifier, the Unique Resource from Sub-Collection access pattern may be used here, in the format: projects/-/androidApps/APP_ID Refer to the AndroidApp [name](../projects.androidApps#AndroidApp.FIELDS.name) field for details about PROJECT_IDENTIFIER and APP_ID values.
        */
      name: String
  )(input: UndeleteAndroidAppRequest): F[Operation] = expectJson[Operation](
    requestWithBody(method = Method.POST, uri = baseUri / "v1beta1" / s"${name}")(input)
  )
  def create(
      /** The resource name of the parent FirebaseProject in which to create an AndroidApp, in the format: projects/PROJECT_IDENTIFIER/androidApps Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      parent: String
  )(input: AndroidApp): F[Operation] =
    expectJson[Operation](
      requestWithBody(
        method = Method.POST,
        uri = baseUri / "v1beta1" / s"${parent}" / "androidApps",
      )(input)
    )
  def remove(
      /** Required. The resource name of the AndroidApp, in the format: projects/ PROJECT_IDENTIFIER/androidApps/APP_ID Since an APP_ID is a unique identifier, the Unique Resource from Sub-Collection access pattern may be used here, in the format: projects/-/androidApps/APP_ID Refer to the AndroidApp [name](../projects.androidApps#AndroidApp.FIELDS.name) field for details about PROJECT_IDENTIFIER and APP_ID values.
        */
      name: String
  )(input: RemoveAndroidAppRequest): F[Operation] = expectJson[Operation](
    requestWithBody(method = Method.POST, uri = baseUri / "v1beta1" / s"${name}")(input)
  )
  def get(
      /** The resource name of the AndroidApp, in the format: projects/ PROJECT_IDENTIFIER/androidApps/APP_ID Since an APP_ID is a unique identifier, the Unique Resource from Sub-Collection access pattern may be used here, in the format: projects/-/androidApps/APP_ID Refer to the `AndroidApp` [`name`](../projects.androidApps#AndroidApp.FIELDS.name) field for details about PROJECT_IDENTIFIER and APP_ID values.
        */
      name: String
  ): F[AndroidApp] =
    expectJson[AndroidApp](request(method = Method.GET, uri = baseUri / "v1beta1" / s"${name}"))
  def patch(
      /** The resource name of the AndroidApp, in the format: projects/ PROJECT_IDENTIFIER/androidApps/APP_ID
        * PROJECT_IDENTIFIER: the parent Project's [`ProjectNumber`](../projects#FirebaseProject.FIELDS.project_number) \*\*\*(recommended)\*\*\* or its [`ProjectId`](../projects#FirebaseProject.FIELDS.project_id). Learn more about using project identifiers in Google's [AIP 2510 standard](https://google.aip.dev/cloud/2510). Note that the value for PROJECT_IDENTIFIER in any response body will be the `ProjectId`.
        * APP_ID: the globally unique, Firebase-assigned identifier for the App (see [`appId`](../projects.androidApps#AndroidApp.FIELDS.app_id)).
        */
      name: String,
      query: ProjectsAndroidAppsClient.PatchParams = ProjectsAndroidAppsClient.PatchParams(),
  )(input: AndroidApp): F[AndroidApp] = expectJson[AndroidApp](
    requestWithBody(
      method = Method.PATCH,
      uri = (baseUri / "v1beta1" / s"${name}")
        .copy(query = Query.fromVector(Vector(List("updateMask" -> query.updateMask).flatMap {
          case (k, v) => v.map(vv => k -> Option(vv))
        }).flatten)),
    )(input)
  )
  def getConfig(
      /** The resource name of the AndroidApp configuration to download, in the format: projects/PROJECT_IDENTIFIER/androidApps/APP_ID/config Since an APP_ID is a unique identifier, the Unique Resource from Sub-Collection access pattern may be used here, in the format: projects/-/androidApps/APP_ID Refer to the `AndroidApp` [`name`](../projects.androidApps#AndroidApp.FIELDS.name) field for details about PROJECT_IDENTIFIER and APP_ID values.
        */
      name: String
  ): F[AndroidAppConfig] = expectJson[AndroidAppConfig](
    request(method = Method.GET, uri = baseUri / "v1beta1" / s"${name}")
  )
  def list(
      /** The resource name of the parent FirebaseProject for which to list each associated AndroidApp, in the format: projects/PROJECT_IDENTIFIER /androidApps Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      parent: String,
      query: ProjectsAndroidAppsClient.ListParams = ProjectsAndroidAppsClient.ListParams(),
  ): F[ListAndroidAppsResponse] = expectJson[ListAndroidAppsResponse](
    request(
      method = Method.GET,
      uri = (baseUri / "v1beta1" / s"${parent}" / "androidApps").copy(query =
        Query.fromVector(
          Vector(
            List("pageSize" -> query.pageSize.map(s => QueryParamEncoder[Int].encode(s).value))
              .flatMap { case (k, v) => v.map(vv => k -> Option(vv)) },
            List("pageToken" -> query.pageToken).flatMap { case (k, v) =>
              v.map(vv => k -> Option(vv))
            },
            List(
              "showDeleted" -> query.showDeleted.map(s =>
                QueryParamEncoder[Boolean].encode(s).value
              )
            ).flatMap { case (k, v) => v.map(vv => k -> Option(vv)) },
          ).flatten
        )
      ),
    )
  )
}
object ProjectsAndroidAppsClient {
  final case class PatchParams(
      /** Specifies which fields of the AndroidApp to update. Note that the following fields are immutable: `name`, `app_id`, `project_id`, and `package_name`. To update `state`, use any of the following endpoints: RemoveAndroidApp or UndeleteAndroidApp.
        */
      updateMask: Option[String] = None
  )
  final case class ListParams(
      /** The maximum number of Apps to return in the response. The server may return fewer than this at its discretion. If no value is specified (or too large a value is specified), then the server will impose its own limit.
        */
      pageSize: Option[Int] = None,
      /** Token returned from a previous call to `ListAndroidApps` indicating where in the set of Apps to resume listing.
        */
      pageToken: Option[String] = None,
      /** Controls whether Apps in the DELETED state should be returned in the response. If not specified, only `ACTIVE` Apps will be returned.
        */
      showDeleted: Option[Boolean] = None,
  )
}
