package googleapis.firebase

import io.circe._

sealed abstract class LocationFeature(val value: String) extends Product with Serializable
object LocationFeature {

  /** Used internally for distinguishing unset values and is not intended for external use.
    */
  case object LOCATION_FEATURE_UNSPECIFIED extends LocationFeature("LOCATION_FEATURE_UNSPECIFIED")

  /** This location supports Cloud Firestore database instances. Google App Engine is available in this location, so it can be a Project's location for default Google Cloud resources.
    */
  case object FIRESTORE extends LocationFeature("FIRESTORE")

  /** This location supports default Cloud Storage buckets. Google App Engine is available in this location, so it can be a Project's location for default Google Cloud resources.
    */
  case object DEFAULT_STORAGE extends LocationFeature("DEFAULT_STORAGE")

  /** Cloud Functions for Firebase is available in this location.
    */
  case object FUNCTIONS extends LocationFeature("FUNCTIONS")
  val values = List(LOCATION_FEATURE_UNSPECIFIED, FIRESTORE, DEFAULT_STORAGE, FUNCTIONS)
  def fromString(input: String): Either[String, LocationFeature] =
    values.find(_.value == input).toRight(s"'$input' was not a valid value for LocationFeature")
  implicit val decoder: Decoder[LocationFeature] = Decoder[String].emap(s => fromString(s))
  implicit val encoder: Encoder[LocationFeature] = Encoder[String].contramap(_.value)
}
