package googleapis.firebase

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.client.Client

class ProjectsWebAppsClient[F[_]: Concurrent](client: Client[F]) extends AbstractClient[F](client) {
  val baseUri = uri"https://firebase.googleapis.com/"
  def undelete(
      /** Required. The resource name of the WebApp, in the format: projects/ PROJECT_IDENTIFIER/webApps/APP_ID Since an APP_ID is a unique identifier, the Unique Resource from Sub-Collection access pattern may be used here, in the format: projects/-/webApps/APP_ID Refer to the WebApp [name](../projects.webApps#WebApp.FIELDS.name) field for details about PROJECT_IDENTIFIER and APP_ID values.
        */
      name: String
  )(input: UndeleteWebAppRequest): F[Operation] = expectJson[Operation](
    requestWithBody(method = Method.POST, uri = baseUri / "v1beta1" / s"${name}")(input)
  )
  def create(
      /** The resource name of the parent FirebaseProject in which to create a WebApp, in the format: projects/PROJECT_IDENTIFIER/webApps Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      parent: String
  )(input: WebApp): F[Operation] = expectJson[Operation](
    requestWithBody(method = Method.POST, uri = baseUri / "v1beta1" / s"${parent}" / "webApps")(
      input
    )
  )
  def remove(
      /** Required. The resource name of the WebApp, in the format: projects/ PROJECT_IDENTIFIER/webApps/APP_ID Since an APP_ID is a unique identifier, the Unique Resource from Sub-Collection access pattern may be used here, in the format: projects/-/webApps/APP_ID Refer to the WebApp [name](../projects.webApps#WebApp.FIELDS.name) field for details about PROJECT_IDENTIFIER and APP_ID values.
        */
      name: String
  )(input: RemoveWebAppRequest): F[Operation] = expectJson[Operation](
    requestWithBody(method = Method.POST, uri = baseUri / "v1beta1" / s"${name}")(input)
  )
  def get(
      /** The resource name of the WebApp, in the format: projects/PROJECT_IDENTIFIER /webApps/APP_ID Since an APP_ID is a unique identifier, the Unique Resource from Sub-Collection access pattern may be used here, in the format: projects/-/webApps/APP_ID Refer to the `WebApp` [`name`](../projects.webApps#WebApp.FIELDS.name) field for details about PROJECT_IDENTIFIER and APP_ID values.
        */
      name: String
  ): F[WebApp] =
    expectJson[WebApp](request(method = Method.GET, uri = baseUri / "v1beta1" / s"${name}"))
  def patch(
      /** The resource name of the WebApp, in the format: projects/PROJECT_IDENTIFIER /webApps/APP_ID
        * PROJECT_IDENTIFIER: the parent Project's [`ProjectNumber`](../projects#FirebaseProject.FIELDS.project_number) \*\*\*(recommended)\*\*\* or its [`ProjectId`](../projects#FirebaseProject.FIELDS.project_id). Learn more about using project identifiers in Google's [AIP 2510 standard](https://google.aip.dev/cloud/2510). Note that the value for PROJECT_IDENTIFIER in any response body will be the `ProjectId`.
        * APP_ID: the globally unique, Firebase-assigned identifier for the App (see [`appId`](../projects.webApps#WebApp.FIELDS.app_id)).
        */
      name: String,
      query: ProjectsWebAppsClient.PatchParams = ProjectsWebAppsClient.PatchParams(),
  )(input: WebApp): F[WebApp] = expectJson[WebApp](
    requestWithBody(
      method = Method.PATCH,
      uri = (baseUri / "v1beta1" / s"${name}").copy(query = Query("updateMask" -> query.updateMask)),
    )(input)
  )
  def getConfig(
      /** The resource name of the WebApp configuration to download, in the format: projects/PROJECT_IDENTIFIER/webApps/APP_ID/config Since an APP_ID is a unique identifier, the Unique Resource from Sub-Collection access pattern may be used here, in the format: projects/-/webApps/APP_ID Refer to the `WebApp` [`name`](../projects.webApps#WebApp.FIELDS.name) field for details about PROJECT_IDENTIFIER and APP_ID values.
        */
      name: String
  ): F[WebAppConfig] =
    expectJson[WebAppConfig](request(method = Method.GET, uri = baseUri / "v1beta1" / s"${name}"))
  def list(
      /** The resource name of the parent FirebaseProject for which to list each associated WebApp, in the format: projects/PROJECT_IDENTIFIER/webApps Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      parent: String,
      query: ProjectsWebAppsClient.ListParams = ProjectsWebAppsClient.ListParams(),
  ): F[ListWebAppsResponse] = expectJson[ListWebAppsResponse](
    request(
      method = Method.GET,
      uri = (baseUri / "v1beta1" / s"${parent}" / "webApps").copy(query =
        Query(
          "pageSize" -> query.pageSize.map(s => QueryParamEncoder[Int].encode(s).value),
          "pageToken" -> query.pageToken,
          "showDeleted" -> query.showDeleted.map(s => QueryParamEncoder[Boolean].encode(s).value),
        )
      ),
    )
  )
}
object ProjectsWebAppsClient {
  final case class PatchParams(
      /** Specifies which fields of the WebApp to update. Note that the following fields are immutable: `name`, `app_id`, and `project_id`. To update `state`, use any of the following endpoints: RemoveWebApp or UndeleteWebApp.
        */
      updateMask: Option[String] = None
  )
  final case class ListParams(
      /** The maximum number of Apps to return in the response. The server may return fewer than this value at its discretion. If no value is specified (or too large a value is specified), then the server will impose its own limit.
        */
      pageSize: Option[Int] = None,
      /** Token returned from a previous call to `ListWebApps` indicating where in the set of Apps to resume listing.
        */
      pageToken: Option[String] = None,
      /** Controls whether Apps in the DELETED state should be returned in the response. If not specified, only `ACTIVE` Apps will be returned.
        */
      showDeleted: Option[Boolean] = None,
  )
}
