package googleapis.firebase

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.client.Client

class ProjectsClient[F[_]: Concurrent](client: Client[F]) extends AbstractClient[F](client) {
  val baseUri = uri"https://firebase.googleapis.com/"
  def addGoogleAnalytics(
      /** The resource name of the FirebaseProject to link to an existing Google Analytics account, in the format: projects/PROJECT_IDENTIFIER Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      parent: String
  )(input: AddGoogleAnalyticsRequest): F[Operation] = expectJson[Operation](
    requestWithBody(method = Method.POST, uri = baseUri / "v1beta1" / s"${parent}")(input)
  )
  def getAnalyticsDetails(
      /** The resource name of the FirebaseProject, in the format: projects/ PROJECT_IDENTIFIER/analyticsDetails Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      name: String
  ): F[AnalyticsDetails] = expectJson[AnalyticsDetails](
    request(method = Method.GET, uri = baseUri / "v1beta1" / s"${name}")
  )
  def searchApps(
      /** The parent FirebaseProject for which to list Apps, in the format: projects/ PROJECT_IDENTIFIER Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      parent: String,
      query: ProjectsClient.SearchAppsParams = ProjectsClient.SearchAppsParams(),
  ): F[SearchFirebaseAppsResponse] = expectJson[SearchFirebaseAppsResponse](
    request(
      method = Method.GET,
      uri = (baseUri / "v1beta1" / s"${parent}").copy(query =
        Query(
          "filter" -> query.filter,
          "pageSize" -> query.pageSize.map(s => QueryParamEncoder[Int].encode(s).value),
          "pageToken" -> query.pageToken,
          "showDeleted" -> query.showDeleted.map(s => QueryParamEncoder[Boolean].encode(s).value),
        )
      ),
    )
  )
  def addFirebase(
      /** The resource name of the GCP `Project` to which Firebase resources will be added, in the format: projects/PROJECT_IDENTIFIER Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values. After calling `AddFirebase`, the unique Project identifiers ( [`projectNumber`](https://cloud.google.com/resource-manager/reference/rest/v1/projects#Project.FIELDS.project_number) and [`projectId`](https://cloud.google.com/resource-manager/reference/rest/v1/projects#Project.FIELDS.project_id)) of the underlying GCP `Project` are also the identifiers of the FirebaseProject.
        */
      project: String
  )(input: AddFirebaseRequest): F[Operation] = expectJson[Operation](
    requestWithBody(method = Method.POST, uri = baseUri / "v1beta1" / s"${project}")(input)
  )
  def removeAnalytics(
      /** The resource name of the FirebaseProject to unlink from its Google Analytics account, in the format: projects/PROJECT_IDENTIFIER Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      parent: String
  )(input: RemoveAnalyticsRequest): F[Empty] = expectJson[Empty](
    requestWithBody(method = Method.POST, uri = baseUri / "v1beta1" / s"${parent}")(input)
  )
  def get(
      /** The resource name of the FirebaseProject, in the format: projects/ PROJECT_IDENTIFIER Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      name: String
  ): F[FirebaseProject] = expectJson[FirebaseProject](
    request(method = Method.GET, uri = baseUri / "v1beta1" / s"${name}")
  )
  def patch(
      /** The resource name of the Project, in the format: projects/PROJECT_IDENTIFIER PROJECT_IDENTIFIER: the Project's [`ProjectNumber`](../projects#FirebaseProject.FIELDS.project_number) \*\*\*(recommended)\*\*\* or its [`ProjectId`](../projects#FirebaseProject.FIELDS.project_id). Learn more about using project identifiers in Google's [AIP 2510 standard](https://google.aip.dev/cloud/2510). Note that the value for PROJECT_IDENTIFIER in any response body will be the `ProjectId`.
        */
      name: String,
      query: ProjectsClient.PatchParams = ProjectsClient.PatchParams(),
  )(input: FirebaseProject): F[FirebaseProject] = expectJson[FirebaseProject](
    requestWithBody(
      method = Method.PATCH,
      uri = (baseUri / "v1beta1" / s"${name}").copy(query = Query("updateMask" -> query.updateMask)),
    )(input)
  )
  def getAdminSdkConfig(
      /** The resource name of the FirebaseProject, in the format: projects/ PROJECT_IDENTIFIER/adminSdkConfig Refer to the `FirebaseProject` [`name`](../projects#FirebaseProject.FIELDS.name) field for details about PROJECT_IDENTIFIER values.
        */
      name: String
  ): F[AdminSdkConfig] =
    expectJson[AdminSdkConfig](request(method = Method.GET, uri = baseUri / "v1beta1" / s"${name}"))
  def list(
      query: ProjectsClient.ListParams = ProjectsClient.ListParams()
  ): F[ListFirebaseProjectsResponse] = expectJson[ListFirebaseProjectsResponse](
    request(
      method = Method.GET,
      uri = (baseUri / "v1beta1" / "projects").copy(query =
        Query(
          "pageSize" -> query.pageSize.map(s => QueryParamEncoder[Int].encode(s).value),
          "pageToken" -> query.pageToken,
          "showDeleted" -> query.showDeleted.map(s => QueryParamEncoder[Boolean].encode(s).value),
        )
      ),
    )
  )
}
object ProjectsClient {
  final case class SearchAppsParams(
      /** A query string compatible with Google's [AIP-160 standard](https://google.aip.dev/160). Use any of the following fields in a query:
        * [`app_id`](../projects/searchApps#FirebaseAppInfo.FIELDS.app_id)
        * [`namespace`](../projects/searchApps#FirebaseAppInfo.FIELDS.namespace)
        * [`platform`](../projects/searchApps#FirebaseAppInfo.FIELDS.platform) This query also supports the following "virtual" fields. These are fields which are not actually part of the returned resource object, but they can be queried as if they are pre-populated with specific values.
        * `sha1_hash` or `sha1_hashes`: This field is considered to be a _repeated_ `string` field, populated with the list of all SHA-1 certificate fingerprints registered with the AndroidApp. This list is empty if the App is not an `AndroidApp`.
        * `sha256_hash` or `sha256_hashes`: This field is considered to be a _repeated_ `string` field, populated with the list of all SHA-256 certificate fingerprints registered with the AndroidApp. This list is empty if the App is not an `AndroidApp`.
        * `app_store_id`: This field is considered to be a _singular_ `string` field, populated with the Apple App Store ID registered with the IosApp. This field is empty if the App is not an `IosApp`.
        * `team_id`: This field is considered to be a _singular_ `string` field, populated with the Apple team ID registered with the IosApp. This field is empty if the App is not an `IosApp`.
        */
      filter: Option[String] = None,
      /** The maximum number of Apps to return in the response. The server may return fewer than this value at its discretion. If no value is specified (or too large a value is specified), then the server will impose its own limit. This value cannot be negative.
        */
      pageSize: Option[Int] = None,
      /** Token returned from a previous call to `SearchFirebaseApps` indicating where in the set of Apps to resume listing.
        */
      pageToken: Option[String] = None,
      /** Controls whether Apps in the DELETED state should be returned. If not specified, only `ACTIVE` Apps will be returned.
        */
      showDeleted: Option[Boolean] = None,
  )
  final case class PatchParams(
      /** Specifies which fields of the FirebaseProject to update. Note that the following fields are immutable: `name`, `project_id`, and `project_number`. To update `state`, use any of the following Google Cloud endpoints: [`projects.delete`](https://cloud.google.com/resource-manager/reference/rest/v1/projects/delete) or [`projects.undelete`](https://cloud.google.com/resource-manager/reference/rest/v1/projects/undelete)
        */
      updateMask: Option[String] = None
  )
  final case class ListParams(
      /** The maximum number of Projects to return in the response. The server may return fewer than this at its discretion. If no value is specified (or too large a value is specified), the server will impose its own limit. This value cannot be negative.
        */
      pageSize: Option[Int] = None,
      /** Token returned from a previous call to `ListFirebaseProjects` indicating where in the set of Projects to resume listing.
        */
      pageToken: Option[String] = None,
      /** Optional. Controls whether Projects in the DELETED state should be returned in the response. If not specified, only `ACTIVE` Projects will be returned.
        */
      showDeleted: Option[Boolean] = None,
  )
}
