package googleapis.bigquery

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.circe._
import org.http4s.client.Client
import io.circe._

class TablesClient[F[_]: Concurrent](client: Client[F]) {
  implicit private def entityEncoder[A: Encoder]: EntityEncoder[F, A] = jsonEncoderOf[F, A]
  implicit private def entityDecoder[A: Decoder]: EntityDecoder[F, A] = jsonOf[F, A]
  val baseUri = uri"https://bigquery.googleapis.com/bigquery/v2"

  def testIamPermissions(
      // REQUIRED: The resource for which the policy detail is being requested. See [Resource names](https://cloud.google.com/apis/design/resource_names) for the appropriate value for this field.
      resource: String
  )(input: TestIamPermissionsRequest): F[Option[TestIamPermissionsResponse]] =
    client.expectOption[TestIamPermissionsResponse](
      Request[F](
        method = Method.POST,
        uri = baseUri / s"${resource}",
      ).withEntity(input)
    )

  def insert(
      // Project ID of the new table
      projectId: String,
      // Dataset ID of the new table
      datasetId: String,
  )(input: Table): F[Option[Table]] =
    client.expectOption[Table](
      Request[F](
        method = Method.POST,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables",
      ).withEntity(input)
    )

  def setIamPolicy(
      // REQUIRED: The resource for which the policy is being specified. See [Resource names](https://cloud.google.com/apis/design/resource_names) for the appropriate value for this field.
      resource: String
  )(input: SetIamPolicyRequest): F[Option[Policy]] =
    client.expectOption[Policy](
      Request[F](
        method = Method.POST,
        uri = baseUri / s"${resource}",
      ).withEntity(input)
    )

  def getIamPolicy(
      // REQUIRED: The resource for which the policy is being requested. See [Resource names](https://cloud.google.com/apis/design/resource_names) for the appropriate value for this field.
      resource: String
  )(input: GetIamPolicyRequest): F[Option[Policy]] =
    client.expectOption[Policy](
      Request[F](
        method = Method.POST,
        uri = baseUri / s"${resource}",
      ).withEntity(input)
    )

  def delete(
      // Project ID of the table to delete
      projectId: String,
      // Dataset ID of the table to delete
      datasetId: String,
      // Table ID of the table to delete
      tableId: String,
  ): F[Status] =
    client.status(
      Request[F](
        method = Method.DELETE,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables" / s"${tableId}",
      )
    )

  def get(
      // Project ID of the requested table
      projectId: String,
      // Dataset ID of the requested table
      datasetId: String,
      // Table ID of the requested table
      tableId: String,
  ): F[Option[Table]] =
    client.expectOption[Table](
      Request[F](
        method = Method.GET,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables" / s"${tableId}",
      )
    )

  def update(
      // Project ID of the table to update
      projectId: String,
      // Dataset ID of the table to update
      datasetId: String,
      // Table ID of the table to update
      tableId: String,
  )(input: Table): F[Option[Table]] =
    client.expectOption[Table](
      Request[F](
        method = Method.PUT,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables" / s"${tableId}",
      ).withEntity(input)
    )

  def patch(
      // Project ID of the table to update
      projectId: String,
      // Dataset ID of the table to update
      datasetId: String,
      // Table ID of the table to update
      tableId: String,
  )(input: Table): F[Option[Table]] =
    client.expectOption[Table](
      Request[F](
        method = Method.PATCH,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables" / s"${tableId}",
      ).withEntity(input)
    )

  def list(
      // Project ID of the tables to list
      projectId: String,
      // Dataset ID of the tables to list
      datasetId: String,
  ): F[Option[TableList]] =
    client.expectOption[TableList](
      Request[F](
        method = Method.GET,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables",
      )
    )

}
