package googleapis.bigquery

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.circe._
import org.http4s.client.Client
import io.circe._

class TabledataClient[F[_]: Concurrent](client: Client[F]) {
  implicit private def entityEncoder[A: Encoder]: EntityEncoder[F, A] = jsonEncoderOf[F, A]
  implicit private def entityDecoder[A: Decoder]: EntityDecoder[F, A] = jsonOf[F, A]
  val baseUri = uri"https://bigquery.googleapis.com/bigquery/v2"

  def insertAll(
      // Project ID of the destination table.
      projectId: String,
      // Dataset ID of the destination table.
      datasetId: String,
      // Table ID of the destination table.
      tableId: String,
  )(input: TableDataInsertAllRequest): F[Option[TableDataInsertAllResponse]] =
    client.expectOption[TableDataInsertAllResponse](
      Request[F](
        method = Method.POST,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables" / s"${tableId}" / "insertAll",
      ).withEntity(input)
    )

  def list(
      // Project ID of the table to read
      projectId: String,
      // Dataset ID of the table to read
      datasetId: String,
      // Table ID of the table to read
      tableId: String,
  ): F[Option[TableDataList]] =
    client.expectOption[TableDataList](
      Request[F](
        method = Method.GET,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables" / s"${tableId}" / "data",
      )
    )

}
