package googleapis.bigquery

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.circe._
import org.http4s.client.Client
import io.circe._

class RowAccessPoliciesClient[F[_]: Concurrent](client: Client[F]) {
  implicit private def entityEncoder[A: Encoder]: EntityEncoder[F, A] = jsonEncoderOf[F, A]
  implicit private def entityDecoder[A: Decoder]: EntityDecoder[F, A] = jsonOf[F, A]
  val baseUri = uri"https://bigquery.googleapis.com/bigquery/v2"

  def getIamPolicy(
      // REQUIRED: The resource for which the policy is being requested. See [Resource names](https://cloud.google.com/apis/design/resource_names) for the appropriate value for this field.
      resource: String
  )(input: GetIamPolicyRequest): F[Option[Policy]] =
    client.expectOption[Policy](
      Request[F](
        method = Method.POST,
        uri = baseUri / s"${resource}",
      ).withEntity(input)
    )

  def list(
      // Required. Project ID of the row access policies to list.
      projectId: String,
      // Required. Dataset ID of row access policies to list.
      datasetId: String,
      // Required. Table ID of the table to list row access policies.
      tableId: String,
  ): F[Option[ListRowAccessPoliciesResponse]] =
    client.expectOption[ListRowAccessPoliciesResponse](
      Request[F](
        method = Method.GET,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "tables" / s"${tableId}" / "rowAccessPolicies",
      )
    )

  def testIamPermissions(
      // REQUIRED: The resource for which the policy detail is being requested. See [Resource names](https://cloud.google.com/apis/design/resource_names) for the appropriate value for this field.
      resource: String
  )(input: TestIamPermissionsRequest): F[Option[TestIamPermissionsResponse]] =
    client.expectOption[TestIamPermissionsResponse](
      Request[F](
        method = Method.POST,
        uri = baseUri / s"${resource}",
      ).withEntity(input)
    )

}
