package googleapis.bigquery

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.circe._
import org.http4s.client.Client
import io.circe._

class RoutinesClient[F[_]: Concurrent](client: Client[F]) {
  implicit private def entityEncoder[A: Encoder]: EntityEncoder[F, A] = jsonEncoderOf[F, A]
  implicit private def entityDecoder[A: Decoder]: EntityDecoder[F, A] = jsonOf[F, A]
  val baseUri = uri"https://bigquery.googleapis.com/bigquery/v2"

  def insert(
      // Required. Project ID of the new routine
      projectId: String,
      // Required. Dataset ID of the new routine
      datasetId: String,
  )(input: Routine): F[Option[Routine]] =
    client.expectOption[Routine](
      Request[F](
        method = Method.POST,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "routines",
      ).withEntity(input)
    )

  def delete(
      // Required. Project ID of the routine to delete
      projectId: String,
      // Required. Dataset ID of the routine to delete
      datasetId: String,
      // Required. Routine ID of the routine to delete
      routineId: String,
  ): F[Status] =
    client.status(
      Request[F](
        method = Method.DELETE,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "routines" / s"${routineId}",
      )
    )

  def get(
      // Required. Project ID of the requested routine
      projectId: String,
      // Required. Dataset ID of the requested routine
      datasetId: String,
      // Required. Routine ID of the requested routine
      routineId: String,
  ): F[Option[Routine]] =
    client.expectOption[Routine](
      Request[F](
        method = Method.GET,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "routines" / s"${routineId}",
      )
    )

  def update(
      // Required. Project ID of the routine to update
      projectId: String,
      // Required. Dataset ID of the routine to update
      datasetId: String,
      // Required. Routine ID of the routine to update
      routineId: String,
  )(input: Routine): F[Option[Routine]] =
    client.expectOption[Routine](
      Request[F](
        method = Method.PUT,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "routines" / s"${routineId}",
      ).withEntity(input)
    )

  def list(
      // Required. Project ID of the routines to list
      projectId: String,
      // Required. Dataset ID of the routines to list
      datasetId: String,
  ): F[Option[ListRoutinesResponse]] =
    client.expectOption[ListRoutinesResponse](
      Request[F](
        method = Method.GET,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "routines",
      )
    )

}
