package googleapis.bigquery

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.circe._
import org.http4s.client.Client
import io.circe._

class ModelsClient[F[_]: Concurrent](client: Client[F]) {
  implicit private def entityEncoder[A: Encoder]: EntityEncoder[F, A] = jsonEncoderOf[F, A]
  implicit private def entityDecoder[A: Decoder]: EntityDecoder[F, A] = jsonOf[F, A]
  val baseUri = uri"https://bigquery.googleapis.com/bigquery/v2"

  def delete(
      // Required. Project ID of the model to delete.
      projectId: String,
      // Required. Dataset ID of the model to delete.
      datasetId: String,
      // Required. Model ID of the model to delete.
      modelId: String,
  ): F[Status] =
    client.status(
      Request[F](
        method = Method.DELETE,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "models" / s"${modelId}",
      )
    )

  def get(
      // Required. Project ID of the requested model.
      projectId: String,
      // Required. Dataset ID of the requested model.
      datasetId: String,
      // Required. Model ID of the requested model.
      modelId: String,
  ): F[Option[Model]] =
    client.expectOption[Model](
      Request[F](
        method = Method.GET,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "models" / s"${modelId}",
      )
    )

  def list(
      // Required. Project ID of the models to list.
      projectId: String,
      // Required. Dataset ID of the models to list.
      datasetId: String,
  ): F[Option[ListModelsResponse]] =
    client.expectOption[ListModelsResponse](
      Request[F](
        method = Method.GET,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "models",
      )
    )

  def patch(
      // Required. Project ID of the model to patch.
      projectId: String,
      // Required. Dataset ID of the model to patch.
      datasetId: String,
      // Required. Model ID of the model to patch.
      modelId: String,
  )(input: Model): F[Option[Model]] =
    client.expectOption[Model](
      Request[F](
        method = Method.PATCH,
        uri =
          baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}" / "models" / s"${modelId}",
      ).withEntity(input)
    )

}
