package googleapis.bigquery

import io.circe._

sealed abstract class ModelModelType(val value: String) extends Product with Serializable
object ModelModelType {
  // Default value.
  case object MODEL_TYPE_UNSPECIFIED extends ModelModelType("MODEL_TYPE_UNSPECIFIED")
  // Linear regression model.
  case object LINEAR_REGRESSION extends ModelModelType("LINEAR_REGRESSION")
  // Logistic regression based classification model.
  case object LOGISTIC_REGRESSION extends ModelModelType("LOGISTIC_REGRESSION")
  // K-means clustering model.
  case object KMEANS extends ModelModelType("KMEANS")
  // Matrix factorization model.
  case object MATRIX_FACTORIZATION extends ModelModelType("MATRIX_FACTORIZATION")
  // DNN classifier model.
  case object DNN_CLASSIFIER extends ModelModelType("DNN_CLASSIFIER")
  // An imported TensorFlow model.
  case object TENSORFLOW extends ModelModelType("TENSORFLOW")
  // DNN regressor model.
  case object DNN_REGRESSOR extends ModelModelType("DNN_REGRESSOR")
  // An imported XGBoost model.
  case object XGBOOST extends ModelModelType("XGBOOST")
  // Boosted tree regressor model.
  case object BOOSTED_TREE_REGRESSOR extends ModelModelType("BOOSTED_TREE_REGRESSOR")
  // Boosted tree classifier model.
  case object BOOSTED_TREE_CLASSIFIER extends ModelModelType("BOOSTED_TREE_CLASSIFIER")
  // ARIMA model.
  case object ARIMA extends ModelModelType("ARIMA")
  // AutoML Tables regression model.
  case object AUTOML_REGRESSOR extends ModelModelType("AUTOML_REGRESSOR")
  // AutoML Tables classification model.
  case object AUTOML_CLASSIFIER extends ModelModelType("AUTOML_CLASSIFIER")
  // Prinpical Component Analysis model.
  case object PCA extends ModelModelType("PCA")
  // Wide-and-deep classifier model.
  case object DNN_LINEAR_COMBINED_CLASSIFIER
      extends ModelModelType("DNN_LINEAR_COMBINED_CLASSIFIER")
  // Wide-and-deep regressor model.
  case object DNN_LINEAR_COMBINED_REGRESSOR extends ModelModelType("DNN_LINEAR_COMBINED_REGRESSOR")
  // Autoencoder model.
  case object AUTOENCODER extends ModelModelType("AUTOENCODER")
  // New name for the ARIMA model.
  case object ARIMA_PLUS extends ModelModelType("ARIMA_PLUS")
  // ARIMA with external regressors.
  case object ARIMA_PLUS_XREG extends ModelModelType("ARIMA_PLUS_XREG")
  // Random forest regressor model.
  case object RANDOM_FOREST_REGRESSOR extends ModelModelType("RANDOM_FOREST_REGRESSOR")
  // Random forest classifier model.
  case object RANDOM_FOREST_CLASSIFIER extends ModelModelType("RANDOM_FOREST_CLASSIFIER")
  // An imported TensorFlow Lite model.
  case object TENSORFLOW_LITE extends ModelModelType("TENSORFLOW_LITE")
  // An imported ONNX model.
  case object ONNX extends ModelModelType("ONNX")
  val values = List(
    MODEL_TYPE_UNSPECIFIED,
    LINEAR_REGRESSION,
    LOGISTIC_REGRESSION,
    KMEANS,
    MATRIX_FACTORIZATION,
    DNN_CLASSIFIER,
    TENSORFLOW,
    DNN_REGRESSOR,
    XGBOOST,
    BOOSTED_TREE_REGRESSOR,
    BOOSTED_TREE_CLASSIFIER,
    ARIMA,
    AUTOML_REGRESSOR,
    AUTOML_CLASSIFIER,
    PCA,
    DNN_LINEAR_COMBINED_CLASSIFIER,
    DNN_LINEAR_COMBINED_REGRESSOR,
    AUTOENCODER,
    ARIMA_PLUS,
    ARIMA_PLUS_XREG,
    RANDOM_FOREST_REGRESSOR,
    RANDOM_FOREST_CLASSIFIER,
    TENSORFLOW_LITE,
    ONNX,
  )
  def fromString(input: String): Either[String, ModelModelType] =
    values.find(_.value == input).toRight(s"'$input' was not a valid value for ModelModelType")
  implicit val decoder: Decoder[ModelModelType] = Decoder[String].emap(s => fromString(s))
  implicit val encoder: Encoder[ModelModelType] = Encoder[String].contramap(_.value)
}
