package googleapis.bigquery

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.circe._
import org.http4s.client.Client
import io.circe._

class JobsClient[F[_]: Concurrent](client: Client[F]) {
  implicit private def entityEncoder[A: Encoder]: EntityEncoder[F, A] = jsonEncoderOf[F, A]
  implicit private def entityDecoder[A: Decoder]: EntityDecoder[F, A] = jsonOf[F, A]
  val baseUri = uri"https://bigquery.googleapis.com/bigquery/v2"

  def cancel(
      // [Required] Project ID of the job to cancel
      projectId: String,
      // [Required] Job ID of the job to cancel
      jobId: String,
  ): F[Option[JobCancelResponse]] =
    client.expectOption[JobCancelResponse](
      Request[F](
        method = Method.POST,
        uri = baseUri / "projects" / s"${projectId}" / "jobs" / s"${jobId}" / "cancel",
      )
    )

  def insert(
      // Project ID of the project that will be billed for the job
      projectId: String
  )(input: Job): F[Option[Job]] =
    client.expectOption[Job](
      Request[F](
        method = Method.POST,
        uri = baseUri / "projects" / s"${projectId}" / "jobs",
      ).withEntity(input)
    )

  def query(
      // Project ID of the project billed for the query
      projectId: String
  )(input: QueryRequest): F[Option[QueryResponse]] =
    client.expectOption[QueryResponse](
      Request[F](
        method = Method.POST,
        uri = baseUri / "projects" / s"${projectId}" / "queries",
      ).withEntity(input)
    )

  def delete(
      // Required. Project ID of the job for which metadata is to be deleted.
      projectId: String,
      // Required. Job ID of the job for which metadata is to be deleted. If this is a parent job which has child jobs, the metadata from all child jobs will be deleted as well. Direct deletion of the metadata of child jobs is not allowed.
      jobId: String,
  ): F[Status] =
    client.status(
      Request[F](
        method = Method.DELETE,
        uri = baseUri / "projects" / s"${projectId}" / "jobs" / s"${jobId}" / "delete",
      )
    )

  def get(
      // [Required] Project ID of the requested job
      projectId: String,
      // [Required] Job ID of the requested job
      jobId: String,
  ): F[Option[Job]] =
    client.expectOption[Job](
      Request[F](
        method = Method.GET,
        uri = baseUri / "projects" / s"${projectId}" / "jobs" / s"${jobId}",
      )
    )

  def list(
      // Project ID of the jobs to list
      projectId: String
  ): F[Option[JobList]] =
    client.expectOption[JobList](
      Request[F](
        method = Method.GET,
        uri = baseUri / "projects" / s"${projectId}" / "jobs",
      )
    )

  def getQueryResults(
      // [Required] Project ID of the query job
      projectId: String,
      // [Required] Job ID of the query job
      jobId: String,
  ): F[Option[GetQueryResultsResponse]] =
    client.expectOption[GetQueryResultsResponse](
      Request[F](
        method = Method.GET,
        uri = baseUri / "projects" / s"${projectId}" / "queries" / s"${jobId}",
      )
    )

}
