package googleapis.bigquery

import cats.effect.Concurrent
import org.http4s._
import org.http4s.implicits._
import org.http4s.circe._
import org.http4s.client.Client
import io.circe._

class DatasetsClient[F[_]: Concurrent](client: Client[F]) {
  implicit private def entityEncoder[A: Encoder]: EntityEncoder[F, A] = jsonEncoderOf[F, A]
  implicit private def entityDecoder[A: Decoder]: EntityDecoder[F, A] = jsonOf[F, A]
  val baseUri = uri"https://bigquery.googleapis.com/bigquery/v2"

  def insert(
      // Project ID of the new dataset
      projectId: String
  )(input: Dataset): F[Option[Dataset]] =
    client.expectOption[Dataset](
      Request[F](
        method = Method.POST,
        uri = baseUri / "projects" / s"${projectId}" / "datasets",
      ).withEntity(input)
    )

  def delete(
      // Project ID of the dataset being deleted
      projectId: String,
      // Dataset ID of dataset being deleted
      datasetId: String,
  ): F[Status] =
    client.status(
      Request[F](
        method = Method.DELETE,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}",
      )
    )

  def get(
      // Project ID of the requested dataset
      projectId: String,
      // Dataset ID of the requested dataset
      datasetId: String,
  ): F[Option[Dataset]] =
    client.expectOption[Dataset](
      Request[F](
        method = Method.GET,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}",
      )
    )

  def update(
      // Project ID of the dataset being updated
      projectId: String,
      // Dataset ID of the dataset being updated
      datasetId: String,
  )(input: Dataset): F[Option[Dataset]] =
    client.expectOption[Dataset](
      Request[F](
        method = Method.PUT,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}",
      ).withEntity(input)
    )

  def patch(
      // Project ID of the dataset being updated
      projectId: String,
      // Dataset ID of the dataset being updated
      datasetId: String,
  )(input: Dataset): F[Option[Dataset]] =
    client.expectOption[Dataset](
      Request[F](
        method = Method.PATCH,
        uri = baseUri / "projects" / s"${projectId}" / "datasets" / s"${datasetId}",
      ).withEntity(input)
    )

  def list(
      // Project ID of the datasets to be listed
      projectId: String
  ): F[Option[DatasetList]] =
    client.expectOption[DatasetList](
      Request[F](
        method = Method.GET,
        uri = baseUri / "projects" / s"${projectId}" / "datasets",
      )
    )

}
