package googleapis.bigquery

import io.circe._

sealed abstract class ArgumentArgumentKind(val value: String) extends Product with Serializable
object ArgumentArgumentKind {
  // Default value.
  case object ARGUMENT_KIND_UNSPECIFIED extends ArgumentArgumentKind("ARGUMENT_KIND_UNSPECIFIED")
  // The argument is a variable with fully specified type, which can be a struct or an array, but not a table.
  case object FIXED_TYPE extends ArgumentArgumentKind("FIXED_TYPE")
  // The argument is any type, including struct or array, but not a table. To be added: FIXED_TABLE, ANY_TABLE
  case object ANY_TYPE extends ArgumentArgumentKind("ANY_TYPE")
  val values = List(ARGUMENT_KIND_UNSPECIFIED, FIXED_TYPE, ANY_TYPE)
  def fromString(input: String): Either[String, ArgumentArgumentKind] = values
    .find(_.value == input)
    .toRight(s"'$input' was not a valid value for ArgumentArgumentKind")
  implicit val decoder: Decoder[ArgumentArgumentKind] = Decoder[String].emap(s => fromString(s))
  implicit val encoder: Encoder[ArgumentArgumentKind] = Encoder[String].contramap(_.value)
}
