package net.guerlab.smart.wx.web.controller;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import net.guerlab.smart.platform.auth.annotation.IgnoreLogin;
import net.guerlab.smart.platform.auth.factory.TokenFactory;
import net.guerlab.smart.platform.auth.factory.TokenFactoryManager;
import net.guerlab.smart.platform.commons.exception.RefreshTokenInvalidException;
import net.guerlab.smart.platform.commons.ip.IpUtils;
import net.guerlab.smart.wx.auth.WxUserContextHandler;
import net.guerlab.smart.wx.core.domain.LoginResponse;
import net.guerlab.smart.wx.core.domain.WxUserDTO;
import net.guerlab.smart.wx.core.entity.IWxUserTokenInfo;
import net.guerlab.smart.wx.core.exception.WxUserInvalidException;
import net.guerlab.smart.wx.service.entity.WxUser;
import net.guerlab.smart.wx.service.service.LoginService;
import net.guerlab.smart.wx.service.service.WxUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestHeader;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * 微信相关-控制面板
 *
 * @author guer
 */
public abstract class AbstractWxControlPanelController<LS extends LoginService> {

    protected WxUserService wxUserService;

    protected LS loginService;

    @IgnoreLogin
    @ApiOperation("登录")
    @GetMapping("/{appId}/login/{code}")
    public LoginResponse login(@ApiParam(value = "应用ID", required = true) @PathVariable String appId,
            @ApiParam(value = "code", required = true) @PathVariable String code, HttpServletRequest request) {
        return loginService.login(appId, code, IpUtils.getIp(request), request.getHeader("Referer"), null);
    }

    @IgnoreLogin
    @ApiOperation("刷新Token")
    @GetMapping("/{appId}/refreshToken")
    public LoginResponse refreshToken(@ApiParam(value = "应用ID", required = true) @PathVariable String appId,
            @ApiParam(value = "refreshToken", required = true) @RequestHeader("refreshToken") String refreshToken,
            HttpServletRequest request) {
        String ip = IpUtils.getIp(request);
        TokenFactory<IWxUserTokenInfo> tokenFactory = TokenFactoryManager.getTokenFactories(IWxUserTokenInfo.class)
                .stream().filter(factory -> factory.acceptRefreshToken(refreshToken) && factory.acceptIp(ip))
                .findFirst().orElseThrow(RefreshTokenInvalidException::new);

        IWxUserTokenInfo info = tokenFactory.parseByRefreshToken(refreshToken);

        if (!Objects.equals(appId, info.getAppId())) {
            throw new RefreshTokenInvalidException();
        }

        return loginService
                .buildLoginResponse(wxUserService.findUser(info.getAppId(), info.getOpenId()), tokenFactory, ip);
    }

    @ApiOperation("获取个人信息")
    @GetMapping
    public WxUserDTO getInfo() {
        return findCurrentWxUser().toDTO();
    }

    protected WxUser findCurrentWxUser() {
        WxUser wxUser = wxUserService.findUser(WxUserContextHandler.getAppId(), WxUserContextHandler.getOpenId());

        if (wxUser == null) {
            throw new WxUserInvalidException();
        }

        return wxUser;
    }

    @Autowired
    public void setWxUserService(WxUserService wxUserService) {
        this.wxUserService = wxUserService;
    }

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    public void setLoginService(LS loginService) {
        this.loginService = loginService;
    }
}
