package net.guerlab.smart.wx.web.controller.user.wx.mp;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.mp.api.WxMpUserService;
import me.chanjar.weixin.mp.bean.result.WxMpUser;
import me.chanjar.weixin.mp.bean.result.WxMpUserList;
import net.guerlab.smart.wx.core.exception.WxAppInvalidException;
import net.guerlab.smart.wx.core.exception.WxMpUserSynchronizingException;
import net.guerlab.smart.wx.service.entity.WxApp;
import net.guerlab.smart.wx.service.entity.WxUser;
import net.guerlab.smart.wx.service.service.WxAppService;
import net.guerlab.smart.wx.service.service.WxMpManagerService;
import net.guerlab.smart.wx.service.service.WxUserService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 微信-公众号-用户管理
 *
 * @author guer
 */
@Slf4j
@Api(tags = "微信-公众号-用户管理")
@RestController("/user/wx/mp/user")
@RequestMapping("/user/wx/mp/user")
public class UserController {

    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(Runtime.getRuntime().availableProcessors(),
            Runtime.getRuntime().availableProcessors() * 2, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(),
            r -> {
                Thread thread = new Thread(r);
                thread.setName("async_wx_mp_user");
                return thread;
            });

    private final Map<String, AtomicBoolean> lockMap = new ConcurrentHashMap<>(8);

    private WxMpManagerService mpManagerService;

    private WxUserService wxUserService;

    private WxAppService wxAppService;

    @ApiOperation("同步用户信息")
    @PostMapping("/{appId}/userSync")
    public void userSync(@ApiParam(value = "应用ID", required = true) @PathVariable String appId) {
        final AtomicBoolean flag = lockMap.computeIfAbsent(appId, s -> new AtomicBoolean());

        if (!flag.compareAndSet(false, true)) {
            throw new WxMpUserSynchronizingException();
        }

        WxApp wxApp = wxAppService.selectByIdOptional(appId).orElseThrow(WxAppInvalidException::new);
        executor.execute(() -> {
            try {
                userSyncThread(appId, wxApp.getAppName());
            } catch (Exception e) {
                log.debug(e.getLocalizedMessage(), e);
            } finally {
                flag.set(false);
            }
        });
    }

    private void userSyncThread(String appId, String appName) throws Exception {
        WxMpUserService userService = mpManagerService.getService(appId).getUserService();

        boolean hasMore = true;
        String nextOpenid = null;

        log.debug("user sync start[appId: {}]", appId);

        while (hasMore) {
            WxMpUserList wxMpUserList = userService.userList(nextOpenid);
            nextOpenid = wxMpUserList.getNextOpenid();
            hasMore = StringUtils.isNotBlank(nextOpenid);
            List<String> openIds = wxMpUserList.getOpenids();

            if (openIds.isEmpty()) {
                break;
            }

            userService.userInfoList(wxMpUserList.getOpenids())
                    .forEach((user) -> this.userSyncHandler(appId, appName, user));
        }

        log.debug("user sync finished[appId: {}]", appId);
    }

    private void userSyncHandler(String appId, String appName, WxMpUser user) {
        WxUser wxUser = wxUserService.findUser(appId, user.getOpenId());

        if (wxUser == null) {
            wxUser = new WxUser();
            wxUser.setOpenId(user.getOpenId());
            wxUser.setAppId(appId);
            wxUser.setUnionId(user.getUnionId());
            wxUser.setAppName(appName);
            wxUser.setUnionId(user.getUnionId());
            wxUser.setAvatarUrl(user.getHeadImgUrl());
            wxUser.setNickName(user.getNickname());

            wxUserService.insertSelective(wxUser);
        } else {
            WxUser update = new WxUser();
            update.setOpenId(wxUser.getOpenId());
            update.setAppId(appId);
            update.setUnionId(user.getUnionId());
            update.setAvatarUrl(user.getHeadImgUrl());
            update.setNickName(user.getNickname());
            update.setVersion(wxUser.getVersion());

            wxUserService.updateSelectiveById(update);
        }
    }

    @Autowired
    public void setMpManagerService(WxMpManagerService mpManagerService) {
        this.mpManagerService = mpManagerService;
    }

    @Autowired
    public void setWxUserService(WxUserService wxUserService) {
        this.wxUserService = wxUserService;
    }

    @Autowired
    public void setWxAppService(WxAppService wxAppService) {
        this.wxAppService = wxAppService;
    }
}
