/*
 * Copyright 2018-2021 guerlab.net and other contributors.
 *
 * Licensed under the GNU LESSER GENERAL PUBLIC LICENSE, Version 3 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.guerlab.smart.wx.service.service.impl;

import lombok.extern.slf4j.Slf4j;
import net.guerlab.commons.collection.CollectionUtil;
import net.guerlab.commons.number.NumberHelper;
import net.guerlab.smart.platform.server.service.BaseServiceImpl;
import net.guerlab.smart.wx.core.exception.AppIdInvalidException;
import net.guerlab.smart.wx.core.exception.OpenIdInvalidException;
import net.guerlab.smart.wx.core.searchparams.WxUserSearchParams;
import net.guerlab.smart.wx.service.domain.RegisterStatistics;
import net.guerlab.smart.wx.service.entity.UserTagMapping;
import net.guerlab.smart.wx.service.entity.WxUser;
import net.guerlab.smart.wx.service.handler.AfterWxUserUpdateHandler;
import net.guerlab.smart.wx.service.mapper.WxUserMapper;
import net.guerlab.smart.wx.service.searchparams.UserTagMappingSearchParams;
import net.guerlab.smart.wx.service.service.UserTagMappingService;
import net.guerlab.smart.wx.service.service.WxUserService;
import net.guerlab.smart.wx.stream.binders.WxAppChangeMessage;
import net.guerlab.smart.wx.stream.binders.WxAppChangeSubscriberChannel;
import net.guerlab.spring.commons.util.SpringApplicationContextUtil;
import net.guerlab.spring.searchparams.SearchParamsUtils;
import net.guerlab.web.result.ListObject;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.stream.annotation.EnableBinding;
import org.springframework.cloud.stream.annotation.StreamListener;
import org.springframework.messaging.Message;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 微信用户服务实现
 *
 * @author guer
 */
@Slf4j
@Service
@EnableBinding({ WxAppChangeSubscriberChannel.class })
public class WxUserServiceImpl extends BaseServiceImpl<WxUser, String, WxUserMapper, WxUserSearchParams> implements WxUserService {

    private static final ThreadPoolExecutor EXECUTOR;

    private UserTagMappingService tagMappingService;

    static {
        int availableProcessors = Runtime.getRuntime().availableProcessors();
        EXECUTOR = new ThreadPoolExecutor(availableProcessors, availableProcessors * 2, 1L, TimeUnit.MINUTES, new LinkedBlockingQueue<>(), r -> {
            Thread thread = new Thread(r);
            thread.setName("wxUser updateBefore");
            return thread;
        }, new ScheduledThreadPoolExecutor.DiscardOldestPolicy());
    }

    @Override
    public ListObject<WxUser> queryPage(WxUserSearchParams searchParams) {
        if (refuseQuery(searchParams)) {
            return ListObject.empty();
        }
        return selectPage(searchParams);
    }

    @Override
    public Collection<WxUser> queryAll(WxUserSearchParams searchParams) {
        if (refuseQuery(searchParams)) {
            return Collections.emptyList();
        }
        return selectAll(searchParams);
    }

    private boolean refuseQuery(WxUserSearchParams searchParams) {
        if (NumberHelper.greaterZero(searchParams.getTagId()) || CollectionUtil.isNotEmpty(searchParams.getTagIds())) {
            UserTagMappingSearchParams userTagMappingSearchParams = new UserTagMappingSearchParams();
            userTagMappingSearchParams.setTagId(searchParams.getTagId());
            userTagMappingSearchParams.setTagIds(searchParams.getTagIds());

            List<String> openIds = CollectionUtil.toList(tagMappingService.selectAll(userTagMappingSearchParams), UserTagMapping::getOpenId);

            if (openIds.isEmpty()) {
                return true;
            }

            searchParams.setOpenIds(openIds);
        }
        return false;
    }

    @Override
    public WxUser findUser(String appId, String openId) {
        if (StringUtils.isAnyBlank(appId, openId)) {
            return null;
        }

        WxUserSearchParams searchParams = new WxUserSearchParams();
        searchParams.setAppId(appId);
        searchParams.setOpenId(openId);

        return selectOne(searchParams);
    }

    @Override
    public void delete(String appId, String openId) {
        String trimAppId = StringUtils.trimToNull(appId);
        String trimOpenId = StringUtils.trimToNull(openId);

        if (trimAppId == null || trimOpenId == null) {
            return;
        }

        WxUserSearchParams searchParams = new WxUserSearchParams();
        searchParams.setAppId(trimAppId);
        searchParams.setOpenId(trimOpenId);

        getBaseMapper().delete(getQueryWrapper(searchParams));
    }

    @Override
    public Collection<RegisterStatistics> selectRegisterStatistics(WxUserSearchParams searchParams) {
        Map<String, Object> params = new HashMap<>(16);
        SearchParamsUtils.handler(searchParams, params);
        try {
            return getBaseMapper().selectRegisterStatistics(params);
        } catch (Exception e) {
            log.debug(e.getLocalizedMessage(), e);
            return Collections.emptyList();
        }
    }

    @Override
    protected void insertBefore(WxUser entity) {
        String appId = StringUtils.trimToNull(entity.getAppId());
        String openId = StringUtils.trimToNull(entity.getOpenId());

        if (appId == null) {
            throw new AppIdInvalidException();
        }
        if (openId == null) {
            throw new OpenIdInvalidException();
        }

        entity.setOpenId(openId);
        entity.setAppId(appId);
        entity.setUnionId(StringUtils.trimToEmpty(entity.getUnionId()));
        if (entity.getActivated() == null) {
            entity.setActivated(false);
        }
    }

    @Override
    protected void updateBefore(WxUser entity) {
        EXECUTOR.execute(
                () -> SpringApplicationContextUtil.getContext().getBeansOfType(AfterWxUserUpdateHandler.class).values().forEach(handler -> handler.afterWxUserUpdateHandler(entity)));
    }

    @StreamListener(WxAppChangeSubscriberChannel.NAME)
    public void wxAppChangeMessageHandler(Message<WxAppChangeMessage> message) {
        WxAppChangeMessage payload = message.getPayload();

        if (payload.isDeleteFlag()) {
            return;
        }

        String appId = StringUtils.trimToNull(payload.getAppId());
        String appName = payload.getApp() == null ? null : StringUtils.trimToNull(payload.getApp().getAppName());

        if (appId == null || appName == null) {
            return;
        }

        WxUser wxUser = new WxUser();
        wxUser.setAppName(appName);

        WxUserSearchParams searchParams = new WxUserSearchParams();
        searchParams.setAppId(appId);

        getBaseMapper().update(wxUser, getQueryWrapper(searchParams));
    }

    @Autowired
    public void setTagMappingService(UserTagMappingService tagMappingService) {
        this.tagMappingService = tagMappingService;
    }
}
